/* ONLY one of these can be defined */

#define MSGB_DRIVER

#if defined(REGB_DRIVER) && defined(MSGB_DRIVER)
#error Must define only ONE of REGB_DRIVER or MSGB_DRIVER
#endif 

#if !defined(REGB_DRIVER) && !defined(MSGB_DRIVER)
#error MUST have either REGB_DRIVER or MSGB_DRIVER defined!
#endif

/* If you have context sensitive error messages then uncomment the following
define.  If you don't know what this means then leave commented out:
#define WANT_CDE_INIT
*/

#include <stdlib.h>		/* prototype for malloc() */
#include <string.h>		/* prototype for strcpy() */
#include <stdio.h>		/* prototype for sprintf() */
#include <malloc.h>
#include <windows.h> 
#include "visa.h"
#include "visatype.h"

#ifdef WIN32
#define _fmemcpy memcpy
#endif

#ifdef REGB_DRIVER
#include "cscpiv.h"		/* includes cscpi and visa calls */
#endif

#include <windows.h>

#include "visa.h"
#include "hpe1445.h"

struct ListStruct {
					 char name[15];
					 int size;
					 double *data;
					 struct ListStruct * next;
					 ViBoolean loaded;
					 char *MarkerList;
};

long setDelay( double val )
{
//Delay should be absolute (e.g. 100 micro sec = 1e-4)

double slice;
__int64 count;


if( !QueryPerformanceFrequency( (LARGE_INTEGER *)&count ) )
	{
		//hdw doens't have high perfomance count so use getickcount
		slice = 1e-3; //units for gettick count
	}
else
	{
		slice = 1.0/count; //Seconds per tick
	}

return (long)(val/slice)+1;

}

void doDelay( long ticks )
{
__int64 startval,tmp;

 if(!QueryPerformanceCounter( (LARGE_INTEGER *)&startval))
	{
		DWORD sval;
		sval = GetTickCount();
		while( GetTickCount()-sval  < (DWORD)ticks) ;
		return; 
	}
 tmp = startval;
 while(tmp-startval < (DWORD)ticks)
 {
	QueryPerformanceCounter( (LARGE_INTEGER *)&tmp);
 }

}


#define hpe1445_MODEL_CODE 418	
#define hpe1445HV_MODEL_CODE 423
#define hpe1445_MANF_ID     4095	/* ??? you must change this if not hp card*/
#define hpe1445_IDN_STRING  "HEWLETT-PACKARD,E1445"	/* ??? you must change this */

#define hpe1445_REV_CODE "A.02.01"  /* Driver Revision */	/* ??? you must change this */

#define hpe1445_ERR_MSG_LENGTH 256  /* size of error message buffer */

/*#define hpe1445_MAX_STAT_HAP 37*/		    /* number of happenings ??? */ 
#define hpe1445_MAX_STAT_HAP 34
#define hpe1445_MAX_STAT_REG 4		    /* number of IEEE 488.2 status registers */

/* this has to match the index of the ESR register in hpe1445_accessInfo[] */
#define hpe1445_ESR_REG_IDX 2

/* this needs to match the index location of hpe1445_USER_ERROR_HANLDER
 * in the hpe1445_statusHap[] array.  Normally, this is zero.
 */
#define hpe1445_USER_ERROR_HANDLER_IDX 0

/* you can remove the SWAP's below if you do not have block IO (aritrary block)
   if you do then you must also remove the routines that use them (_cmd_arr functions)
*/
#define SWAP_FLOAT64(dest) \
 { unsigned char    src[8];                                        \
      *((double *)src) = *((double *)dest);                        \
      ((unsigned char *)(dest))[0] =  ((unsigned char*)(src))[7];  \
      ((unsigned char *)(dest))[1] =  ((unsigned char*)(src))[6];  \
      ((unsigned char *)(dest))[2] =  ((unsigned char*)(src))[5];  \
      ((unsigned char *)(dest))[3] =  ((unsigned char*)(src))[4];  \
      ((unsigned char *)(dest))[4] =  ((unsigned char*)(src))[3];  \
      ((unsigned char *)(dest))[5] =  ((unsigned char*)(src))[2];  \
      ((unsigned char *)(dest))[6] =  ((unsigned char*)(src))[1];  \
      ((unsigned char *)(dest))[7] =  ((unsigned char*)(src))[0];  \
 }

#define SWAP_FLOAT32(dest) \
 { unsigned char    src[4];                                        \
      *((float *)src) = *((float *)dest);                        \
      ((unsigned char *)(dest))[0] =  ((unsigned char*)(src))[3];  \
      ((unsigned char *)(dest))[1] =  ((unsigned char*)(src))[2];  \
      ((unsigned char *)(dest))[2] =  ((unsigned char*)(src))[1];  \
      ((unsigned char *)(dest))[3] =  ((unsigned char*)(src))[0];  \
 }

#define SWAP_32(dest) \
 { unsigned char    src[4];                                        \
      *((long *)src) = *((long *)dest);                        \
      ((unsigned char *)(dest))[0] =  ((unsigned char*)(src))[3];  \
      ((unsigned char *)(dest))[1] =  ((unsigned char*)(src))[2];  \
      ((unsigned char *)(dest))[2] =  ((unsigned char*)(src))[1];  \
      ((unsigned char *)(dest))[3] =  ((unsigned char*)(src))[0];  \
 }

#define SWAP_16(dest) \
 { unsigned char    src[2];                                        \
      *((int *)src) = *((int *)dest);                        \
      ((unsigned char *)(dest))[0] =  ((unsigned char*)(src))[1];  \
      ((unsigned char *)(dest))[1] =  ((unsigned char*)(src))[0];  \
 }
/*=============================================================== 
 *
 *  All messages are stored in this area to aid in localization 
 *
 *=============================================================== 
 */


#define hpe1445_MSG_VI_OPEN_ERR 				\
	"vi was zero.  Was the hpe1445_init() successful?"

#define hpe1445_MSG_CONDITION					\
	"condition"
	/* hpe1445_statCond_Q() */

#define hpe1445_MSG_EVENT						\
	"event"	
	/* hpe1445_statEvent_Q() */

#define hpe1445_MSG_EVENT_HDLR_INSTALLED				\
	"event handler is already installed for event happening"
	/* hpe1445_statEvent_Q() */

#define hpe1445_MSG_EVENT_HDLR_INST2				\
	"Only 1 handler can be installed at a time."	
	/* hpe1445_statEvent_Q() */


#define hpe1445_MSG_INVALID_HAPPENING				\
	"is not a valid happening."
	/* hpe1445_statCond_Q() */
	/* hpe1445_statEven_Q() */
	/* hpe1445_statEvenHdlr() */
	/* hpe1445_statEvenHdlr_Q() */

#define hpe1445_MSG_NOT_QUERIABLE					\
	"is not queriable."	
	/* hpe1445_statCond_Q() */
	/* hpe1445_statEven_Q() */


#define hpe1445_MSG_IN_FUNCTION					\
	"in function" 		
	/* hpe1445_error_message() */

#define hpe1445_MSG_INVALID_STATUS					\
  	"Parameter 2 is invalid"				\
	"in function hpe1445_error_message()."
	/* hpe1445_error_message() */

#define hpe1445_MSG_INVALID_STATUS_VALUE				\
	"is not a valid viStatus value."
	/* hpe1445_error_message() */

#define  hpe1445_MSG_INVALID_VI					\
  	"Parameter 1 is invalid"				\
	" in function hpe1445_error_message()"			\
	".  Using an inactive ViSession may cause this error."	\
	"  Was the instrument driver closed prematurely?"
	/* hpe1445_message_query() */

#define hpe1445_MSG_NO_ERRORS					\
	"No Errors"
	/* hpe1445_error_message() */

#define hpe1445_MSG_SELF_TEST_FAILED 				\
	"Self test failed." 	
	/* hpe1445_self_test() */

#define hpe1445_MSG_SELF_TEST_PASSED 				\
	"Self test passed."
	/* hpe1445_self_test() */

#define hpe1445_MSG_ERR_SEGNAME				\
	"Segment not found.  (Probably 'hpe1445_creatSeg' was not called)"

#define hpe1445_MSG_BOOLEAN   "Expected 0 or 1; Got %hd"

#define hpe1445_MSG_REAL   "Expected %lg to %lg; Got %lg"
   
#define hpe1445_MSG_INT   "Expected %hd to %hd; Got %hd"

#define hpe1445_MSG_LONG   "Expected %ld to %ld; Got %ld"

#define hpe1445_MSG_LOOKUP "Error converting string response to integer"

#define hpe1445_MSG_NO_MATCH "Could not match string %s"

/* 
 * static error message 
 */

#define VI_ERROR_PARAMETER1_MSG				\
	"Parameter 1 is invalid"

#define VI_ERROR_PARAMETER2_MSG				\
	"Parameter 2 is invalid"

#define VI_ERROR_PARAMETER3_MSG				\
	"Parameter 3 is invalid"

#define VI_ERROR_PARAMETER4_MSG				\
	"Parameter 4 is invalid"

#define VI_ERROR_PARAMETER5_MSG				\
	"Parameter 5 is invalid"

#define VI_ERROR_PARAMETER6_MSG				\
	"Parameter 6 is invalid"

#define VI_ERROR_PARAMETER7_MSG				\
	"Parameter 7 is invalid"

#define VI_ERROR_PARAMETER8_MSG				\
	"Parameter 8 is invalid"

#define VI_ERROR_PARAMETER9_MSG				\
	"Parameter 9 is invalid"

#define VI_ERROR_PARAMETER10_MSG			\
	"Parameter 10 is invalid"

#define VI_ERROR_PARAMETER11_MSG			\
	"Parameter 11 is invalid"

#define VI_ERROR_PARAMETER12_MSG			\
	"Parameter 12 is invalid"

#define VI_ERROR_PARAMETER13_MSG			\
	"Parameter 13 is invalid"

#define VI_ERROR_PARAMETER14_MSG			\
	"Parameter 14 is invalid"

#define VI_ERROR_PARAMETER15_MSG			\
	"Parameter 15 is invalid"

#define VI_ERROR_PARAMETER16_MSG			\
	"Parameter 16 is invalid"

#define VI_ERROR_PARAMETER17_MSG			\
	"Parameter 17 is invalid"

#define VI_ERROR_PARAMETER18_MSG			\
	"Parameter 18 is invalid"

#define VI_ERROR_FAIL_ID_QUERY_MSG				\
	"Instrument IDN does not match."

#define INSTR_ERROR_INV_SESSION_MSG 				\
	"ViSession (parmeter 1) was not created by this driver"

#define INSTR_ERROR_NULL_PTR_MSG				\
	"NULL pointer detected"

#define INSTR_ERROR_RESET_FAILED_MSG				\
	"reset failed"

#define INSTR_ERROR_UNEXPECTED_MSG 				\
	"An unexpected error occurred"

#define INSTR_ERROR_DETECTED_MSG			\
	"Instrument Error Detected, call hpe1445_error_query()."

#define INSTR_ERROR_NOTFOUND_MSG			\
	"Instrument was not found."

#define INSTR_ERROR_NOBEGIN_MSG				 \
	"coupleBegin either errored or was not called or couple buffer was empty."

#define INSTR_ERROR_LOOKUP_MSG   				\
	"String not found in table"

/*================================================================*/

/* don't check the debug pointer all the time!*/
#ifdef DEBUG
#define hpe1445_DEBUG_CHK_THIS( vi, thisPtr) 			\
	/* check for NULL user data */				\
	if( 0 == thisPtr)					\
	{							\
 		hpe1445_LOG_STATUS(                             	\
		  vi, NULL, hpe1445_INSTR_ERROR_INV_SESSION );	\
	}							\
	{							\
		ViSession defRM;				\
								\
		/* This should never fail */			\
		errStatus = viGetAttribute( vi,                 \
			VI_ATTR_RM_SESSION, &defRM);		\
		if( VI_SUCCESS > errStatus )			\
		{						\
 			hpe1445_LOG_STATUS(				\
			  vi, NULL, hpe1445_INSTR_ERROR_UNEXPECTED );	\
		}						\
		if( defRM != thisPtr->defRMSession)		\
		{						\
 			hpe1445_LOG_STATUS(				\
			  vi, NULL, hpe1445_INSTR_ERROR_INV_SESSION );	\
		}						\
	}
#else
#define hpe1445_DEBUG_CHK_THIS( vi, thisPtr) 	
#endif

#ifdef WANT_CDE_INIT
#define hpe1445_CDE_INIT( funcname)  				\
	strcpy(thisPtr->errFuncName, funcname);			\
	thisPtr->errNumber = VI_SUCCESS;			\
	thisPtr->errMessage[0] = 0;

#define hpe1445_CDE_MESSAGE( message ) 	 			\
	strcpy(thisPtr->errMessage, message)
#else
#define hpe1445_CDE_INIT( funcname) 				\
	thisPtr->errNumber = VI_SUCCESS;			\

#define hpe1445_CDE_MESSAGE( message ) 	 			\
	strcpy(thisPtr->errMessage, message)
#endif



struct hpe1445_globals
{
	ViSession			defRMSession;

	ViStatus			errNumber;
	char 				errFuncName[40];
	char				errMessage[160];

	ViBoolean			errQueryDetect;

	ViBoolean			e1406; /* 1 if gpib to 1406*/

        /* Setting this to VI_TRUE,
         * will prevent I/O from occuring in an SRQ
         * This is needed, because VTL 3.0 doesn't provide
         * an atomic write/read operations.  Don't want
         * to do I/O if the driver is in the middle of
         * a read from the instrument.
         */
        ViBoolean                       blockSrqIO;
        ViInt32                         countSrqIO;
		long				myDelay;
		char * CoupleBuffer;
		int  CoupleSize;
		struct ListStruct * ListHead;
		struct ListStruct * ListTail;
		
        
};
/* add the following to the globals data structure */

/* Note: for debugging, you can add __FILE__, __LINE__ as parameters
 * to hpe1445_LOG_STATUS, and ViString filename, and ViInt32 lineNumber to
 * hpe1445_statusUpdate() in order to determine exactly where an error
 * occured in a driver.
 */
#define hpe1445_LOG_STATUS( vi, thisPtr, status ) 	\
  return hpe1445_statusUpdate( vi, thisPtr, status)

/* declare this here since it is called by statusUpdate */
static void hpe1445_srqTraverse(ViSession vi, ViInt32 eventReg);

ViStatus hpe1445_statusUpdate(
#ifdef MSGB_DRIVER
ViSession vi,
#else
INST_PARAM(vi,"",REGISTER),
#endif
struct hpe1445_globals *thisPtr,ViStatus s) 

#ifdef MSGB_DRIVER
{   
 	ViUInt32 rc;         
 	ViStatus errStatus;
 	char lc[20];
 	ViInt16 eventQ;
 	
	if( thisPtr && thisPtr->errQueryDetect && 
		s != VI_ERROR_TMO/* don't access if already timed out!*/)
		{   
			errStatus = viWrite(vi,"*ESR?",5,&rc);
			if( errStatus < VI_SUCCESS) return VI_ERROR_SYSTEM_ERROR;
			
			errStatus = viRead(vi,lc,20,&rc);
			doDelay( thisPtr->myDelay);
			if( errStatus < VI_SUCCESS) return VI_ERROR_SYSTEM_ERROR;
			
			eventQ=atoi(lc);
			
			if( (  0x04 /* Query Error */
                      | 0x08 /* Device Dependent Error */
                      | 0x10 /* Execution Error */
                      | 0x20 /* Command Error */
                    ) & eventQ )
				return hpe1445_INSTR_ERROR_DETECTED;
		}
 	return s;
}
#endif            

#ifdef REGB_DRIVER   
{
	ViInt32  eventQ;
	ViInt32 rc;
	char lc[20];
    	ViStatus errStatus;
    
	if (!thisPtr) return s;

	/* Check if user wants to query the instrument state.
         * This is only done, if the vi is valid and 
         * no other errors has occured.
         */
	if( thisPtr && thisPtr->errQueryDetect &&
		s != VI_ERROR_TMO/* don't access if already timed out!*/)
    	{
		/* assume IEEE 488.2 Instrument and query standard
                 * status event register for a parser error
                 */	
		if (thisPtr->e1406)
		{
			thisPtr->blockSrqIO = VI_TRUE;

			errStatus = viWrite(vi, "*ESR?",5,&rc);
			if( errStatus < VI_SUCCESS) return VI_ERROR_SYSTEM_ERROR;
			
			errStatus = viRead(vi,lc,20,&rc);
			if( errStatus < VI_SUCCESS) return VI_ERROR_SYSTEM_ERROR;
			
			eventQ = atoi( lc );
			
		} 
		else
		{
			INST_QUERY(vi, "*esr?\n", "%ld%*t\n", &eventQ);
		}
		
		if( (  0x04 /* Query Error */
                      | 0x08 /* Device Dependent Error */
                      | 0x10 /* Execution Error */
                      | 0x20 /* Command Error */
                    ) & eventQ )
				return hpe1445_INSTR_ERROR_DETECTED;
	} 
	return s;
}
#endif

/* 
 * Error Message Structures
 */

struct instrErrStruct
{
	ViStatus errStatus;
	ViString errMessage;
};

const static struct instrErrStruct instrErrMsgTable[] =
{
        { VI_ERROR_PARAMETER1,	VI_ERROR_PARAMETER1_MSG },
        { VI_ERROR_PARAMETER2,	VI_ERROR_PARAMETER2_MSG },
        { VI_ERROR_PARAMETER3,	VI_ERROR_PARAMETER3_MSG },
        { VI_ERROR_PARAMETER4,	VI_ERROR_PARAMETER4_MSG },
        { VI_ERROR_PARAMETER5,	VI_ERROR_PARAMETER5_MSG },
        { VI_ERROR_PARAMETER6,	VI_ERROR_PARAMETER6_MSG },
        { VI_ERROR_PARAMETER7,	VI_ERROR_PARAMETER7_MSG },
        { VI_ERROR_PARAMETER8,	VI_ERROR_PARAMETER8_MSG },
        { hpe1445_INSTR_ERROR_PARAMETER9,	VI_ERROR_PARAMETER9_MSG },
        { hpe1445_INSTR_ERROR_PARAMETER10,	VI_ERROR_PARAMETER10_MSG },
        { hpe1445_INSTR_ERROR_PARAMETER11,	VI_ERROR_PARAMETER11_MSG },
        { hpe1445_INSTR_ERROR_PARAMETER12,	VI_ERROR_PARAMETER12_MSG },
        { hpe1445_INSTR_ERROR_PARAMETER13,	VI_ERROR_PARAMETER13_MSG },
        { hpe1445_INSTR_ERROR_PARAMETER14,	VI_ERROR_PARAMETER14_MSG },
        { hpe1445_INSTR_ERROR_PARAMETER15,	VI_ERROR_PARAMETER15_MSG },
        { hpe1445_INSTR_ERROR_PARAMETER16,	VI_ERROR_PARAMETER16_MSG },
        { hpe1445_INSTR_ERROR_PARAMETER17,	VI_ERROR_PARAMETER17_MSG },
        { hpe1445_INSTR_ERROR_PARAMETER18,	VI_ERROR_PARAMETER18_MSG },
        { VI_ERROR_FAIL_ID_QUERY,	VI_ERROR_FAIL_ID_QUERY_MSG },

	{ hpe1445_INSTR_ERROR_INV_SESSION,	INSTR_ERROR_INV_SESSION_MSG },
        { hpe1445_INSTR_ERROR_NULL_PTR,	INSTR_ERROR_NULL_PTR_MSG },
        { hpe1445_INSTR_ERROR_RESET_FAILED,INSTR_ERROR_RESET_FAILED_MSG },
        { hpe1445_INSTR_ERROR_UNEXPECTED,	INSTR_ERROR_UNEXPECTED_MSG },
	{ hpe1445_INSTR_ERROR_DETECTED,	INSTR_ERROR_DETECTED_MSG },
	
	{ hpe1445_ERR_NOTFOUND,	INSTR_ERROR_NOTFOUND_MSG },
	{ hpe1445_ERR_NOBEGIN,	INSTR_ERROR_NOBEGIN_MSG },
	{hpe1445_ERR_SEGNAME, hpe1445_MSG_ERR_SEGNAME},

	{ hpe1445_INSTR_ERROR_LOOKUP,	INSTR_ERROR_LOOKUP_MSG },

};

/* macros for testing parameters */
#define hpe1445_CHK_BOOLEAN( my_val, err ) if( hpe1445_chk_boolean( thisPtr, my_val) ) hpe1445_LOG_STATUS( vi, thisPtr, err);

static ViBoolean hpe1445_chk_boolean(
  struct hpe1445_globals *thisPtr,
  ViBoolean my_val)
{
   char message[hpe1445_ERR_MSG_LENGTH];
   if( (my_val != VI_TRUE) && (my_val != VI_FALSE) )
   {
      /* true = parameter is invalid */
      sprintf(message, hpe1445_MSG_BOOLEAN, my_val);	
      hpe1445_CDE_MESSAGE(message);				
      /* true = parameter is invalid */
      return VI_TRUE;
   }

   /* false = okay */
   return VI_FALSE;
}


#define hpe1445_CHK_REAL_RANGE( my_val, min, max, err ) if( hpe1445_chk_real_range( thisPtr, my_val, min, max) ) hpe1445_LOG_STATUS( vi, thisPtr, err);

static ViBoolean hpe1445_chk_real_range( 
  struct hpe1445_globals *thisPtr,
  ViReal64 my_val,
  ViReal64 min,
  ViReal64 max)
{
   char message[hpe1445_ERR_MSG_LENGTH];

   if ( ( my_val < min ) || (my_val > max) )  			
   {								
      sprintf(message, hpe1445_MSG_REAL, min, max, my_val);	
      hpe1445_CDE_MESSAGE(message);				
      /* true = parameter is invalid */
      return VI_TRUE;
   }
   return VI_FALSE;
} 
   
#define hpe1445_CHK_INT_RANGE( my_val, min, max, err ) if( hpe1445_chk_int_range( thisPtr, my_val, min, max) ) hpe1445_LOG_STATUS( vi, thisPtr, err);

static ViBoolean hpe1445_chk_int_range( 
  struct hpe1445_globals *thisPtr,
  ViInt16 my_val,
  ViInt16 min,
  ViInt16 max)
{
   char message[hpe1445_ERR_MSG_LENGTH];

   if ( ( my_val < min ) || (my_val > max) )  			
   {								
      sprintf(message, hpe1445_MSG_INT, min, max, my_val);	
      hpe1445_CDE_MESSAGE(message);				
      /* true = parameter is invalid */
      return VI_TRUE;
   }
   return VI_FALSE;
} 
   
   
#define hpe1445_CHK_LONG_RANGE( my_val, min, max, err ) if( hpe1445_chk_long_range( thisPtr, my_val, min, max) ) hpe1445_LOG_STATUS( vi, thisPtr, err);

static ViBoolean hpe1445_chk_long_range( 
  struct hpe1445_globals *thisPtr,
  ViInt32 my_val,
  ViInt32 min,
  ViInt32 max)
{
   char message[hpe1445_ERR_MSG_LENGTH];

   if ( ( my_val < min ) || (my_val > max) )  			
   {								
      sprintf(message, hpe1445_MSG_LONG, min, max, my_val);	
      hpe1445_CDE_MESSAGE(message);				
      /* true = parameter is invalid */
      return VI_TRUE;
   }
   return VI_FALSE;
} 
   
   
#define hpe1445_CHK_ENUM( my_val, limit, err ) if( hpe1445_chk_enum( thisPtr, my_val, limit) ) hpe1445_LOG_STATUS( vi, thisPtr, err);

/* utility routine which searches for a string in an array of strings. */
/* This is used by the CHK_ENUM macro */
static ViBoolean hpe1445_chk_enum (
  struct hpe1445_globals *thisPtr,
  ViInt16 my_val,
  ViInt16 limit)
{
    char message[hpe1445_ERR_MSG_LENGTH];

    if ( ( my_val < 0 ) || (my_val > limit) )  			
    {								
        sprintf(message, hpe1445_MSG_INT, 0, limit, my_val);	
        hpe1445_CDE_MESSAGE(message);				
        /* true = parameter is invalid */
        return VI_TRUE;
    }

    return VI_FALSE;
}



/*  ==========================================================================  
     This function searches an array of strings for a specific string and     
     returns its index.  If successful, a VI_SUCCESS is returned, 
     else hpe1445_INSTR_ERROR_LOOKUP is returned.
    ======================================================================== */
ViStatus hpe1445_findIndex (struct hpe1445_globals *thisPtr,
			const char * const array_of_strings[],
			  /*last entry in array must be 0 */
			const char *string, /* string read from instrument */
			ViPInt16 index) /* result index */
{
    ViInt16 i;
    ViInt16 my_len;
    char search_str[20];
    char message[80];

    strcpy(search_str, string);

  /* get rid of newline if present in string */
  /* needed because %s includes newline in some VTL's */
    my_len = strlen(search_str);
    if (search_str[my_len - 1] == '\n')
       search_str[my_len - 1] = '\0';

    for (i = 0; array_of_strings[i]; i++)
    {
	if (!strcmp (array_of_strings[i], search_str))
	{
	    *index = i;
	    return VI_SUCCESS;
	}
    }

  /* if we got here, we did not find it */
    sprintf(message, hpe1445_MSG_NO_MATCH, search_str);	
    hpe1445_CDE_MESSAGE(message);				
    return hpe1445_INSTR_ERROR_LOOKUP;
}

#ifdef REGB_DRIVER
static const ViStatus cscpi_error_to_vi_error[] = {
	VI_ERROR_SYSTEM_ERROR,
	VI_ERROR_SYSTEM_ERROR, /* should never happen*/
	VI_ERROR_FAIL_ID_QUERY, /* 2 wrong card type*/
	VI_ERROR_ALLOC,		/* 3 out of memory*/
	VI_ERROR_INV_RSRC_NAME, /* 4 bad address*/
	VI_ERROR_RSRC_NFOUND,	/* 5 open failed*/
	VI_ERROR_SYSTEM_ERROR, /*  6 open go system error*/
	VI_ERROR_SYSTEM_ERROR, /*  7 internal errors*/
	VI_ERROR_SYSTEM_ERROR, /*  8 internal errors*/
	VI_ERROR_SYSTEM_ERROR, /*  9 internal errors*/
	VI_ERROR_SYSTEM_ERROR, /*  10 internal errors*/
	VI_ERROR_SYSTEM_ERROR, /*  11 internal errors*/
	VI_ERROR_SYSTEM_ERROR, /*  12 internal errors*/
	VI_ERROR_SYSTEM_ERROR, /*  13 internal errors*/
	VI_ERROR_SYSTEM_ERROR, /*  14 internal errors*/
};
#endif

/* returns the globals pointer */

#ifdef MSGB_DRIVER
#define GetGlobals(vi,thisPtr)\
{\
	errStatus = viGetAttribute( vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);\
	if( VI_SUCCESS > errStatus)\
	{\
	hpe1445_LOG_STATUS( vi, NULL, errStatus);\
	}\
}
#endif
#ifdef REGB_DRIVER
#define GetGlobals( vi,thisPtr)\
{\
	ViUInt16 intf;\
	viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);\
	if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)\
	{\
	errStatus = viGetAttribute( vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);\
	}\
	else\
	{\
	errStatus = instr_getdata( vi,  (ViAddr) &thisPtr);\
	}\
	if( VI_SUCCESS > errStatus)\
	{\
	hpe1445_LOG_STATUS( vi, NULL, errStatus);\
	}\
}
#endif

/****************************************************************************
hpe1445_init
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | InstrDesc                                         ViRsrc      IN
  |   ---------------------------------------------------------------------
  |  | The Instrument Description.
  |  |
  |  | Examples: VXI::5, GPIB-VXI::128::INSTR
   ------------------------------------------------------------------------
  | id_query                                          ViBoolean   IN
  |   ---------------------------------------------------------------------
  |  | if( VI_TRUE) Perform In-System Verification.
  |  | if(VI_FALSE) Do not perform In-System Verification
   ------------------------------------------------------------------------
  | do_reset                                          ViBoolean   IN
  |   ---------------------------------------------------------------------
  |  | IF( VI_TRUE) Perform Reset Operation.
  |  | if(VI_FALSE) Do not perform Reset operation
   ------------------------------------------------------------------------
  | vi                                                ViPSession  OUT
  |   ---------------------------------------------------------------------
  |  | Instrument Handle. This is VI_NULL if an error occurred
  |  | during the init.

*****************************************************************************/

ViStatus _VI_FUNC hpe1445_init(ViRsrc InstrDesc,ViBoolean id_query,ViBoolean do_reset,
#ifdef MSGB_DRIVER
ViPSession vi)
#endif
#ifdef REGB_DRIVER
INST_PARAM(*vi,"",REGISTER))
#endif
#ifdef MSGB_DRIVER
{
	struct hpe1445_globals *thisPtr;
	ViStatus errStatus;
	ViSession defRM;

	ViUInt16 manfId;
	ViUInt16 modelCode;

	*vi = VI_NULL;

	/* Find the Default Resource Manager */
	errStatus = viOpenDefaultRM( &defRM);
	if( VI_SUCCESS > errStatus)
	{
		/* Errors: VI_ERROR_SYSTEM_ERROR 
		 *         VI_ERROR_ALLOC
		 */
		hpe1445_LOG_STATUS( *vi, NULL, errStatus);
	}

	/* Open a session to the instrument */
	errStatus = viOpen( defRM, InstrDesc, VI_NULL, VI_NULL, vi);
	if( VI_SUCCESS > errStatus)
	{	
		viClose( defRM);
		/* Errors: VI_ERROR_NSUP_OPER
		 *         VI_ERROR_INV_RSRC_NAME
		 *         VI_ERROR_INV_ACC_MODE
		 *         VI_ERROR_RSRC_NFOUND
		 *         VI_ERROR_ALLOC
		 */
		*vi = VI_NULL;
		hpe1445_LOG_STATUS( *vi, NULL, errStatus);
	}

	/* get memory for instance specific globals */
	thisPtr = (struct hpe1445_globals *)malloc(sizeof( struct hpe1445_globals) );
	if( 0 == thisPtr)
	{
		viClose( defRM);	/* also closes vi session */
		*vi = VI_NULL;
		hpe1445_LOG_STATUS(*vi, NULL, VI_ERROR_ALLOC);
	}
	thisPtr->CoupleBuffer = NULL;
	thisPtr->ListHead = thisPtr->ListTail = NULL;
	
	/* associate memory with session, should not fail because
	 *   session is valid; and attribute is defined, supported,
	 *   and writable.
	 */
	errStatus = viSetAttribute( 
	              *vi, 
		      VI_ATTR_USER_DATA, 
	              (ViAttrState)thisPtr); 
	if( VI_SUCCESS > errStatus)
	{
		viClose( *vi);
		viClose( defRM);	/* also closes vi session */
		*vi = VI_NULL;
		hpe1445_LOG_STATUS(*vi, NULL, errStatus);
	}

	/* initialize instance globals */
	thisPtr->defRMSession = defRM;
        thisPtr->errNumber = VI_SUCCESS;
	thisPtr->errFuncName[0] = 0;
	thisPtr->errMessage[0] = 0;
	thisPtr->errQueryDetect = VI_FALSE;
	thisPtr->blockSrqIO = VI_FALSE;
	thisPtr->countSrqIO = 0;
	thisPtr->myDelay = setDelay( 1e-4 );


	if( VI_TRUE == do_reset ) 
	{
		/* dev clr andcall the reset function to reset the instrument */
		if( viClear(*vi) < VI_SUCCESS || hpe1445_reset(*vi) < VI_SUCCESS )
		{
			/* ignore any errors in PREFIX_close */
			hpe1445_close( *vi);
                        *vi=VI_NULL;
			hpe1445_LOG_STATUS( *vi, NULL, 
			  hpe1445_INSTR_ERROR_RESET_FAILED);
		}
		Sleep(100);
	}

	if( VI_TRUE == id_query ) 
	{
		ViUInt16 intf;
		char     idn_buf[256];
		
		/* find the interface type */
		if( errStatus = viGetAttribute( *vi, VI_ATTR_INTF_TYPE, &intf) )
		{	
			/* Errors: VI_ERROR_NSUP_ATTR */

			/* ignore any errors in PREFIX_close */
			hpe1445_close( *vi);
                        *vi=VI_NULL;
	
			hpe1445_LOG_STATUS( *vi, NULL, errStatus);
		}

		switch( intf)
		{
			case VI_INTF_GPIB:
				if (
				    viPrintf(*vi, "*IDN?\n") < VI_SUCCESS ||
				    viScanf(*vi, "%t", idn_buf) < VI_SUCCESS ||
				    /* check for a idn match */
				    strncmp(idn_buf, hpe1445_IDN_STRING, 
					    strlen(hpe1445_IDN_STRING))   
				    )
                		{   
					/* ignore any errors in PREFIX_close */
					hpe1445_close( *vi);
                                        *vi=VI_NULL;
					hpe1445_LOG_STATUS( *vi, NULL,
					  VI_ERROR_FAIL_ID_QUERY);
                		}
				doDelay( thisPtr->myDelay);
				break;

			case VI_INTF_VXI:
			case VI_INTF_GPIB_VXI:
	
				/* find the VXI manfacturer's ID */
				errStatus = viGetAttribute( *vi, 
				    VI_ATTR_MANF_ID, &manfId);
				if( VI_SUCCESS > errStatus)
				{	
					/* Errors: VI_ERROR_NSUP_ATTR */
	
					/* ignore any errors in PREFIX_close */
					hpe1445_close( *vi);
                                        *vi=VI_NULL;
	
					hpe1445_LOG_STATUS( *vi, NULL,
					  errStatus);
				}

				/* find the instrument's model code */
				errStatus = viGetAttribute( *vi,VI_ATTR_MODEL_CODE,(ViPAttrState)(&modelCode)); 
				if( VI_SUCCESS > errStatus)
				{	
					/* Errors: VI_ERROR_NSUP_ATTR */
					/* Note: this should never happen 
					 *   with a VXI instrument
					 */

					/* ignore any errors in PREFIX_close */
					hpe1445_close( *vi);
                                        *vi=VI_NULL;
					hpe1445_LOG_STATUS( *vi, NULL, errStatus);
				}

				if( (manfId != hpe1445_MANF_ID) || 
				    ( modelCode != hpe1445_MODEL_CODE && modelCode != hpe1445HV_MODEL_CODE) )
				{
					/* ignore any errors in PREFIX_close */
					hpe1445_close( *vi);
                                        *vi = VI_NULL;

					hpe1445_LOG_STATUS( *vi, NULL,
					  VI_ERROR_FAIL_ID_QUERY);
				}

				break;

			default:
				/* ignore any errors in PREFIX_close */
				hpe1445_close( *vi);
				*vi = VI_NULL;
				hpe1445_LOG_STATUS( *vi, NULL,
				  hpe1445_INSTR_ERROR_UNEXPECTED);

		}	

	} /* if - id_query */

	hpe1445_LOG_STATUS( *vi, thisPtr, VI_SUCCESS);
}
/* MSGB_DRIVER for _init*/
#endif 

#ifdef REGB_DRIVER
{
	struct hpe1445_globals *thisPtr;
	ViStatus errStatus;
	ViSession defRM;
	ViUInt16 intf;

	ViUInt16 manfId;
	ViUInt16 modelCode;

	*vi = VI_NULL;

	/* Find the Default Resource Manager */
	errStatus = viOpenDefaultRM( &defRM);
	if( VI_SUCCESS > errStatus)
	{
		/* Errors: VI_ERROR_SYSTEM_ERROR 
		 *         VI_ERROR_ALLOC
		 */
		hpe1445_LOG_STATUS( *vi, NULL, errStatus);
	}

	/* Open a session to the instrument */
	errStatus = viOpen( defRM, InstrDesc, VI_NULL, VI_NULL, vi);
	if( VI_SUCCESS > errStatus)
	{
		/* can't open using vtl, try register based open*/
		intf = VI_INTF_VXI;
		goto gpib_check_done;
	}
	
	if( errStatus = viGetAttribute( *vi, VI_ATTR_INTF_TYPE, &intf) )
	{
		viClose( *vi);
		viClose( defRM);
		*vi = VI_NULL;
		hpe1445_LOG_STATUS( *vi, NULL, errStatus);
	}

	/* check for 1406 internal driver by reading *IDN? of 0 and
		*IDN? of secondary for this address*/
	if (intf == VI_INTF_GPIB_VXI)
	{
		char gpibdesc[64];
		ViUInt16 num, primary, secondary;
		ViSession vi1406;
		char idn_buf[256];

		/* Can't use VI_ATTR_INTF_NUM since
		/* HP VTL 3.0 has a defect which returns 7 instead of 0*/
		/* Read the number from the descriptor instead, assuming
			it's GPIB-VXIn.
		/* Set the GPIB board number is the same had the GPIB-VXI 
			board number*/
		if (memcmp(InstrDesc, "GPIB-VXI", 8) || 
			InstrDesc[8] < '0' || InstrDesc[8] > '9')
			num = 0;
		else
			num = (ViUInt16) atoi(InstrDesc+8);
		viGetAttribute(*vi, VI_ATTR_GPIB_PRIMARY_ADDR, &primary);
		sprintf(gpibdesc, "GPIB%d::%d::0", (int)num,	(int)primary);
		errStatus = viOpen( defRM, gpibdesc, VI_NULL, VI_NULL, 
					&vi1406);
		if (errStatus < VI_SUCCESS)
			goto gpib_check_done;

		/* try sending an IDN? to 1406 */
		viSetAttribute(vi1406, VI_ATTR_TMO_VALUE, 1000); 
		errStatus = viClear(vi1406);
		if (errStatus < VI_SUCCESS)
		{
			viClose(vi1406);
			goto gpib_check_done;
		}

		errStatus = viPrintf(vi1406, "*IDN?\n");
		if (errStatus < VI_SUCCESS)
		{
			viClose(vi1406);
			goto gpib_check_done;
		}

		/* it worked, try getting response and check it*/
		errStatus = viScanf( vi1406, "%t", idn_buf);
		viClose(vi1406);
		if (errStatus < VI_SUCCESS || 
				memcmp(idn_buf, "HEWLETT-PACKARD,E140", 20))
			goto gpib_check_done;

		/* try sending an IDN? to the card drivers*/
		viSetAttribute(*vi, VI_ATTR_TMO_VALUE, 1000); 
		errStatus = viClear(*vi);
		if (errStatus < VI_SUCCESS)
			goto gpib_check_done;

		errStatus = viPrintf(*vi, "*IDN?\n");
		if (errStatus < VI_SUCCESS)
			goto gpib_check_done;

		/* it worked, try getting response*/
		errStatus = viScanf( *vi, "%t", idn_buf);
		if (errStatus < VI_SUCCESS)
			goto gpib_check_done;

		doDelay( thisPtr->myDelay);
		/* got an answer so switch vi to gpib mode */
		viGetAttribute(*vi, VI_ATTR_GPIB_SECONDARY_ADDR, &secondary);
		sprintf(gpibdesc, "GPIB%d::%d::%d", (int)num, (int)primary, (int)secondary);
		viClose(*vi);
		errStatus = viOpen( defRM, gpibdesc, VI_NULL, VI_NULL, vi);
		if( errStatus < VI_SUCCESS)
		{ /* this should never happen*/	
			viClose( defRM);
			*vi = VI_NULL;
			hpe1445_LOG_STATUS( *vi, NULL, errStatus);
		}
		intf = VI_INTF_GPIB;
	}

gpib_check_done:

	/* get memory for instance specific globals */
	thisPtr = (struct hpe1445_globals *)malloc(sizeof( struct hpe1445_globals) );
	if( 0 == thisPtr)
	{
		viClose( defRM);	/* also closes vi session */
		*vi = VI_NULL;
		hpe1445_LOG_STATUS(*vi, NULL, VI_ERROR_ALLOC);
	}

	if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
	{
		/* associate memory with session, should not fail because
		 *   session is valid; and attribute is defined, supported,
		 *   and writable.
		 */
		errStatus = viSetAttribute( 
		              *vi, 
			      VI_ATTR_USER_DATA, 
		              (ViAttrState)thisPtr); 
		if( VI_SUCCESS > errStatus)
		{
			viClose( *vi);
			viClose( defRM);	/* also closes vi session */
			*vi = VI_NULL;
			hpe1445_LOG_STATUS(*vi, NULL, errStatus);
		}
		thisPtr->e1406 = 1;
	}
	else
	{
		/* reg based drivers open their own sessions*/
		viClose( defRM);	/* also closes vi session */

		/* start pos if not started*/
		INST_STARTUP();

		INST_OPEN(*vi, InstrDesc);

		if (!*vi)
		{
			if (cscpi_open_error >= 0 && cscpi_open_error <= 14)
				hpe1445_LOG_STATUS(*vi, NULL,
			       	cscpi_error_to_vi_error[cscpi_open_error]);
			else
				hpe1445_LOG_STATUS(*vi, NULL, VI_ERROR_SYSTEM_ERROR);
		}

		/* associate memory with session */
		instr_setdata(*vi, thisPtr); 
		thisPtr->e1406 = 0;
	}
	/* initialize instance globals */
	viGetAttribute( *vi, VI_ATTR_RM_SESSION, &thisPtr->defRMSession);
        thisPtr->errNumber = VI_SUCCESS;
	thisPtr->errFuncName[0] = 0;
	thisPtr->errMessage[0] = 0;
	thisPtr->errQueryDetect = VI_FALSE;
        thisPtr->blockSrqIO = VI_FALSE;
        thisPtr->countSrqIO = 0;


	if( VI_TRUE == do_reset ) 
	{
		/* call the reset function to reset the instrument */
		if( hpe1445_dcl(*vi)<VI_SUCCESS || hpe1445_reset(*vi)< VI_SUCCESS )
		{
			/* ignore any errors in hpe1445_close */
			hpe1445_close( *vi);
                        *vi=VI_NULL;
			hpe1445_LOG_STATUS( *vi, NULL, 
			  hpe1445_INSTR_ERROR_RESET_FAILED);
		}
	}

	if( VI_TRUE == id_query ) 
	{
		char     idn_buf[256];

		/* find the interface type */
		if( errStatus = viGetAttribute( *vi, VI_ATTR_INTF_TYPE, &intf) )
		{	
			/* Errors: VI_ERROR_NSUP_ATTR */

			/* ignore any errors in hpe1445_close */
			hpe1445_close( *vi);
                        *vi=VI_NULL;
	
			hpe1445_LOG_STATUS( *vi, NULL, errStatus);
		}

		switch( intf)
		{
			case VI_INTF_GPIB:
				if (
				    viClear(*vi) < VI_SUCCESS ||
				    viPrintf(*vi, "*IDN?\n") < VI_SUCCESS ||
				    viScanf(*vi, "%t", idn_buf) < VI_SUCCESS ||
				    /* check for a idn match */
				    strncmp(idn_buf, hpe1445_IDN_STRING, 
					    strlen(hpe1445_IDN_STRING))   
				    )
                		{   
					/* ignore any errors in hpe1445_close */
					hpe1445_close( *vi);
                                        *vi=VI_NULL;
					hpe1445_LOG_STATUS( *vi, NULL,
					  VI_ERROR_FAIL_ID_QUERY);
                		}
				    doDelay( thisPtr->myDelay );
				break;

			case VI_INTF_VXI:
			case VI_INTF_GPIB_VXI:
	
				/* find the VXI manfacturer's ID */
				errStatus = viGetAttribute( *vi, 
				    VI_ATTR_MANF_ID, &manfId);
				if( VI_SUCCESS > errStatus)
				{	
					/* Errors: VI_ERROR_NSUP_ATTR */
	
					/* ignore any errors in hpe1445_close */
					hpe1445_close( *vi);
                                        *vi=VI_NULL;
	
					hpe1445_LOG_STATUS( *vi, NULL,
					  errStatus);
				}

				/* find the instrument's model code */
				errStatus = viGetAttribute( *vi,VI_ATTR_MODEL_CODE,(ViPAttrState)(&modelCode)); 
				if( VI_SUCCESS > errStatus)
				{	
					/* Errors: VI_ERROR_NSUP_ATTR */
					/* Note: this should never happen 
					 *   with a VXI instrument
					 */

					/* ignore any errors in hpe1445_close */
					hpe1445_close( *vi);
                                        *vi=VI_NULL;
					hpe1445_LOG_STATUS( *vi, NULL, errStatus);
				}

				if( (manfId != hpe1445_MANF_ID) || 
				    ( modelCode != hpe1445_MODEL_CODE && modelCode != hpe1445HV_MODEL_CODE) )
				{
					/* ignore any errors in hpe1445_close */
					hpe1445_close( *vi);
                                        *vi = VI_NULL;

					hpe1445_LOG_STATUS( *vi, NULL,
					  VI_ERROR_FAIL_ID_QUERY);
				}

				break;

			default:
				/* ignore any errors in hpe1445_close */
				hpe1445_close( *vi);
				*vi = VI_NULL;
				hpe1445_LOG_STATUS( *vi, NULL,
				  hpe1445_INSTR_ERROR_UNEXPECTED);

		}	

	} /* if - id_query */
                                   
	hpe1445_LOG_STATUS( *vi, thisPtr, VI_SUCCESS);
}
#endif  /* REGB_DRIVER for _init */ 

/****************************************************************************
hpe1445_close
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1445_init()

*****************************************************************************/
ViStatus _VI_FUNC hpe1445_close( 
#ifdef MSGB_DRIVER
ViSession vi)
#endif
#ifdef REGB_DRIVER
INST_PARAM(vi,"",REGISTER))
#endif

#ifdef MSGB_DRIVER
{
	struct hpe1445_globals *thisPtr;
	ViStatus errStatus;
	ViSession defRM;

	GetGlobals( vi,thisPtr );
	hpe1445_DEBUG_CHK_THIS( vi, thisPtr);

	/* retrieve Resource Management session */
	defRM = thisPtr->defRMSession;

	/* free memory */
	if( thisPtr)		
	{
		struct ListStruct * p;
		/* make sure there is something to free */
		if( thisPtr->CoupleBuffer) free(thisPtr->CoupleBuffer);
		p=thisPtr->ListHead;
		while(p)
		{
		  struct ListStruct * tmp;
		  free(p->data);
		  if( p->MarkerList ) free(p->MarkerList);
		  tmp=p;
		  p=p->next;
		  free(tmp);
		}
		free( thisPtr);
	}	

	
	/* close the vi and RM sessions */
	return viClose( defRM);
}
/* MSGB_DRIVER for _close */
#endif

#ifdef REGB_DRIVER
{
	struct hpe1445_globals *thisPtr;
	ViStatus errStatus;
	ViSession defRM;

	GetGlobals(vi,thisPtr)
	hpe1445_DEBUG_CHK_THIS( vi, thisPtr);

	if (thisPtr->e1406)
	{
		/* retrieve Resource Management session */
		defRM = thisPtr->defRMSession;
	}

	/* free memory */
	if( thisPtr)		
	{
		/* make sure there is something to free */
		free( thisPtr);
	}	

	/* close the session */
	if (thisPtr->e1406)
	{
		/* close the vi and RM sessions */
		return viClose( defRM);
	}
	else
	{
		INST_CLOSE(vi);
	}

	return VI_SUCCESS;
}
/* REGB_DRIVER for _close*/
#endif


/****************************************************************************
hpe1445_reset
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1445_init()

*****************************************************************************/
ViStatus _VI_FUNC hpe1445_reset( 
#ifdef MSGB_DRIVER
ViSession vi)
#endif
#ifdef REGB_DRIVER
INST_PARAM(vi,"",REGISTER))
#endif

#ifdef MSGB_DRIVER
{
	struct hpe1445_globals *thisPtr;
	ViStatus errStatus;

	GetGlobals(vi,thisPtr)
	hpe1445_DEBUG_CHK_THIS( vi, thisPtr);

	errStatus = viPrintf( vi, "*RST\n");
	if( VI_SUCCESS > errStatus)
	{
		
		hpe1445_LOG_STATUS( vi, thisPtr, errStatus);
	}

	hpe1445_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
}
/* MSGB_DRIVER for _reset */
#endif

#ifdef REGB_DRIVER
{
	struct hpe1445_globals *thisPtr;
	ViStatus errStatus;

	GetGlobals(vi,thisPtr)
	hpe1445_DEBUG_CHK_THIS( vi, thisPtr);

	if (thisPtr->e1406)
	{
		errStatus = viPrintf( vi, "*RST\n");
	}
	else
	{
		INST_SEND( vi, "*RST\n");
	}
	if( VI_SUCCESS > errStatus)
	{
		
		hpe1445_LOG_STATUS( vi, thisPtr, errStatus);
	}

	hpe1445_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
}
/* REGB_DRIVER for _reset */
#endif


/****************************************************************************
hpe1445_self_test
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1445_init()
   ------------------------------------------------------------------------
  | test_result                                       ViPInt16    OUT
  |   ---------------------------------------------------------------------
  |  | Numeric result from self-test operation
  |  |
  |  | 0 = no error ( test passed)
  |  | 1 = failure
   ------------------------------------------------------------------------
  | test_message                                      ViChar _VI_FAR []OUT
  |   ---------------------------------------------------------------------
  |  | Self-test status message.  This is limited to 256 characters.

*****************************************************************************/

ViStatus _VI_FUNC hpe1445_self_test(
#ifdef MSGB_DRIVER 
ViSession vi,
#else
INST_PARAM(vi, "", REGISTER),
#endif
ViPInt16 test_result,ViPString test_message)

#ifdef MSGB_DRIVER
{
	struct hpe1445_globals *thisPtr;
	ViStatus errStatus;

	/* initialize output parameters */
	*test_result = -1; 
	test_message[0] = 0; 

	GetGlobals(vi,thisPtr)
	hpe1445_DEBUG_CHK_THIS( vi, thisPtr);

	thisPtr->blockSrqIO = VI_TRUE;

	errStatus = viPrintf( vi, "*TST?\n");
	if( VI_SUCCESS > errStatus)
	{	
		
		hpe1445_LOG_STATUS( vi, thisPtr, errStatus);
	}

	errStatus = viScanf( vi, "%hd%*t", test_result);
	doDelay( thisPtr->myDelay );
	if( VI_SUCCESS > errStatus)
	{
		*test_result = -1; 
		
		hpe1445_LOG_STATUS( vi, thisPtr, errStatus);
	}

	switch( *test_result)
	{
           case 0:
              sprintf(test_message, hpe1445_MSG_SELF_TEST_PASSED);
              break;

	      /*??? add instrument specific cases here*/

           default:
              sprintf(test_message, hpe1445_MSG_SELF_TEST_FAILED);
              break;
        }

	
	hpe1445_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
}

/* MSGB_DRIVER for _self_test */
#endif

#ifdef REGB_DRIVER
{
	struct hpe1445_globals *thisPtr;
	ViStatus errStatus;

	/* initialize output parameters */
	*test_result = -1; 
	test_message[0] = 0; 

	GetGlobals(vi,thisPtr)
	hpe1445_DEBUG_CHK_THIS( vi, thisPtr);

	if (thisPtr->e1406)
	{
		thisPtr->blockSrqIO = VI_TRUE;

		errStatus = viPrintf( vi, "*TST?\n");
		if( VI_SUCCESS > errStatus)
		{	
		
			hpe1445_LOG_STATUS( vi, thisPtr, errStatus);
		}

		errStatus = viScanf( vi, "%hd%*t", test_result);
		doDelay( thisPtr->myDelay );

	}
	else
	{
		INST_QUERY( vi, "*TST?\n","%hd\n", test_result);
	}
	if( VI_SUCCESS > errStatus)
	{
		*test_result = -1; 
		
		hpe1445_LOG_STATUS( vi, thisPtr, errStatus);
	}

	switch( *test_result)
	{
           case 0:
              sprintf(test_message, hpe1445_MSG_SELF_TEST_PASSED);
              break;

	      /*??? add instrument specific cases here*/

           default:
              sprintf(test_message, hpe1445_MSG_SELF_TEST_FAILED);
              break;
        }

	
	hpe1445_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
}
/* REGB_DRIVER for _self_test */
#endif

/****************************************************************************
hpe1445_error_query
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1445_init()
   ------------------------------------------------------------------------
  | error_number                                      ViPInt32    OUT
  |   ---------------------------------------------------------------------
  |  | Instruments error code
   ------------------------------------------------------------------------
  | error_message                                     ViChar _VI_FAR []OUT
  |   ---------------------------------------------------------------------
  |  | Instrument's error message.  This is limited to 256 characters.

*****************************************************************************/
ViStatus _VI_FUNC hpe1445_error_query(
#ifdef MSGB_DRIVER
ViSession vi,
#else
INST_PARAM(vi, "", REGISTER),
#endif
ViPInt32 error_number, ViPString error_message)

#ifdef MSGB_DRIVER
{
	struct hpe1445_globals *thisPtr;
	ViStatus errStatus;

	/* initialize output parameters */
	*error_number = -1; 
	error_message[0] = 0; 

	GetGlobals(vi,thisPtr)
	hpe1445_DEBUG_CHK_THIS( vi, thisPtr);

	thisPtr->blockSrqIO = VI_TRUE;

        errStatus = viPrintf( vi, "SYST:ERR?\n");
	if( VI_SUCCESS > errStatus)
	{
		
		hpe1445_LOG_STATUS( vi, thisPtr, errStatus);
	}

	/* get the error number */
	errStatus = viScanf( vi, "%ld,%t", error_number, error_message);
	doDelay( thisPtr->myDelay );

	/* check for error during the scan */
        if( VI_SUCCESS > errStatus)
	{
		*error_number = -1; 
		error_message[0] = 0; 
		hpe1445_LOG_STATUS( vi, thisPtr, errStatus);
	}

	/* get rid of extra LF at the end of the error_message */
	error_message[strlen(error_message)-1] = 0;

	hpe1445_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
}
/* MSGB_DRIVER for _error_query */
#endif

#ifdef REGB_DRIVER
{
	struct hpe1445_globals *thisPtr;
	ViStatus errStatus;

	/* initialize output parameters */
	*error_number = -1; 
	error_message[0] = 0; 

	GetGlobals(vi,thisPtr)
	hpe1445_DEBUG_CHK_THIS( vi, thisPtr);


	/* get the error number */
	if (thisPtr->e1406)
	{
		thisPtr->blockSrqIO = VI_TRUE;

	        errStatus = viPrintf( vi, "SYST:ERR?\n");
		if( VI_SUCCESS > errStatus)
		{

			hpe1445_LOG_STATUS( vi, thisPtr, errStatus);
		}

		/* get the error number */
		errStatus = viScanf( vi, "%ld,%t", error_number, error_message);
		/* check for error during the scan */
        	if( VI_SUCCESS > errStatus)
		{
			*error_number = -1; 
			error_message[0] = 0; 
			hpe1445_LOG_STATUS( vi, thisPtr, errStatus);
		}

		/* get rid of extra LF at the end of the error_message */
		error_message[strlen(error_message)-1] = 0;
	}
	else
	{
        	INST_QUERY( vi, "SYST:ERR?\n", "%ld,%t", error_number, error_message);
	}

	hpe1445_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
}
/* REGB_DRIVER for _error_query */
#endif

/****************************************************************************
hpe1445_error_message
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1445_init().
  |  | This may be VI_NULL.
   ------------------------------------------------------------------------
  | error_number                                      ViStatus    IN
  |   ---------------------------------------------------------------------
  |  | The error return value from an instrument driver function
   ------------------------------------------------------------------------
  | message                                           ViChar _VI_FAR []OUT
  |   ---------------------------------------------------------------------
  |  | Error message string.  This is limited to 256 characters.

*****************************************************************************/

ViStatus _VI_FUNC hpe1445_error_message(
#ifdef MSGB_DRIVER
ViSession vi,
#else
INST_PARAM(vi, "", REGISTER),
#endif
ViStatus error_number,ViChar _VI_FAR message[])
/* works for either kind of driver */
{
	struct hpe1445_globals *thisPtr;
	ViStatus errStatus;  
	ViInt32 idx;

	/* initialize output parameters */
	message[0] = 0;

	thisPtr = NULL;

	/* try to find a thisPtr */
	if( VI_NULL != vi)
	{
		GetGlobals(vi,thisPtr)
		hpe1445_DEBUG_CHK_THIS( vi, thisPtr);
	} 

	if( VI_SUCCESS == error_number)
	{
		sprintf( message, hpe1445_MSG_NO_ERRORS);
		hpe1445_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
	}

	/* return the static error message */
	for(idx=0; 
	    idx < (sizeof instrErrMsgTable / 
	           sizeof( struct instrErrStruct));
	    idx++)
	{
		/* check for a matching error number */
		if( instrErrMsgTable[idx].errStatus == error_number)
		{
			if( (thisPtr) &&
			    (thisPtr->errNumber == error_number))
			{
				/* context dependent error
				 * message is available.
				 */
				sprintf( message,
				  "%s " hpe1445_MSG_IN_FUNCTION " %s() %s",
		          	  instrErrMsgTable[idx].errMessage,
				  thisPtr->errFuncName,
				  thisPtr->errMessage);
			}
			else
			{
				/* No context dependent eror 
				 * message available so copy 
				 * the static error message
				 */
				strcpy( message,
		          	  instrErrMsgTable[idx].errMessage);

			}
            
			hpe1445_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
		}
	}


	/* try VTL/VISA to see if it exists there 
	 *
	 * At this point vi is either VALID or VI_NULL
	 */
	errStatus = viStatusDesc(vi, error_number, message);
	if( VI_SUCCESS == errStatus)
	{
		/* check for a context dependent error message */
		if( (thisPtr) &&
		    (thisPtr->errNumber == error_number))
		{
			/* context dependent error
			 * message is available.
			 */
			strcat( message, " ");
			strcat( message, hpe1445_MSG_IN_FUNCTION);
			strcat( message, " ");
			strcat( message, thisPtr->errFuncName);
			strcat( message, "() ");
			strcat( message, thisPtr->errMessage);
		}

		/* VTL found an error message, so return success */
		hpe1445_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
	}

	/* if we have a VI_NULL, then we need to return a error message */
	if( VI_NULL == vi)
	{
		strcpy(message, hpe1445_MSG_VI_OPEN_ERR);
		hpe1445_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
	} 

	/* user passed in a invalid status */
	sprintf( message, 
	  hpe1445_MSG_INVALID_STATUS
	  "  %ld"
	  hpe1445_MSG_INVALID_STATUS_VALUE
	  , (long)error_number );
	
	hpe1445_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER2);
}


/****************************************************************************
hpe1445_revision_query
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession      IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1445_init()
   ------------------------------------------------------------------------
  | driver_rev                                ViChar _VI_FAR []      OUT
  |   ---------------------------------------------------------------------
  |  | Instrument driver revision.  This is limited to 256 characters.
   ------------------------------------------------------------------------
  | instr_rev                                 ViChar _VI_FAR []      OUT
  |   ---------------------------------------------------------------------
  |  | Instrument firmware revision.  This is limited to 256 characters.

*****************************************************************************/

ViStatus _VI_FUNC hpe1445_revision_query(
#ifdef MSGB_DRIVER
ViSession vi,
#endif
#ifdef REGB_DRIVER
INST_PARAM(vi, "", REGISTER),
#endif
 ViChar _VI_FAR driver_rev[],
 ViChar _VI_FAR instr_rev[])
#ifdef MSGB_DRIVER
{
	struct hpe1445_globals *thisPtr;
	ViStatus errStatus;
        char temp_str[256];		/* temp hold for instr rev string */
        char *last_comma;		/* last comma in *IDN string */
	ViInt32 cnt;

	/* initialize output parameters */
        driver_rev[0] = 0; 
	instr_rev[0] = 0; 

	GetGlobals(vi,thisPtr)
	hpe1445_DEBUG_CHK_THIS( vi, thisPtr);

	sprintf( driver_rev, "%s", hpe1445_REV_CODE);

	thisPtr->blockSrqIO = VI_TRUE;

	if( errStatus = viPrintf( vi, "*IDN?\n"))
	{
        	driver_rev[0] = 0; 
		 
		hpe1445_LOG_STATUS( vi, thisPtr, errStatus); 
	}

	if( errStatus = viRead( vi, temp_str,256,&cnt))
	{
        	driver_rev[0] = 0; 
		instr_rev[0] = 0; 
		 
		hpe1445_LOG_STATUS( vi, thisPtr, errStatus); 
	}
        doDelay( thisPtr->myDelay );
		temp_str[cnt]=0;
        last_comma = strrchr(temp_str,',');
	/* error and exit if last comma not found */
        if (!last_comma) 
        {
           instr_rev[0] = 0;
	   hpe1445_CDE_MESSAGE("no last comma found in IDN string" ); 
	   hpe1445_LOG_STATUS( vi, thisPtr, hpe1445_INSTR_ERROR_UNEXPECTED); 
        }

        strcpy(instr_rev, last_comma+1);

		if( instr_rev[strlen(instr_rev)-1] == '\n') 
			instr_rev[strlen(instr_rev)-1]=0;

	
	hpe1445_LOG_STATUS( vi, thisPtr, VI_SUCCESS); 
}
/* MSGB_DRIVER for _rev_query */
#endif

#ifdef REGB_DRIVER
{
	struct hpe1445_globals *thisPtr;
	ViStatus errStatus;
        char temp_str[256];		/* temp hold for instr rev string */
        char *last_comma;		/* last comma in *IDN string */

	/* initialize output parameters */
        driver_rev[0] = 0; 
	instr_rev[0] = 0; 

	GetGlobals(vi,thisPtr)
	hpe1445_DEBUG_CHK_THIS( vi, thisPtr);

	sprintf( driver_rev, "%s", hpe1445_REV_CODE);

	if (thisPtr->e1406)
	{
		thisPtr->blockSrqIO = VI_TRUE;

		if( errStatus = viPrintf( vi, "*IDN?\n"))
		{
        		driver_rev[0] = 0; 

			hpe1445_LOG_STATUS( vi, thisPtr, errStatus); 
		}

		if( errStatus = viScanf( vi, "%s%*t", temp_str))
		{
        		driver_rev[0] = 0; 
			instr_rev[0] = 0; 
		 
			hpe1445_LOG_STATUS( vi, thisPtr, errStatus); 
		}
	}
	else
	{
		INST_QUERY( vi, "*IDN?\n", "%s", temp_str);
	}
	
        last_comma = strrchr(temp_str,',');
	/* error and exit if last comma not found */
        if (!last_comma) 
        {
           instr_rev[0] = 0;
	   hpe1445_CDE_MESSAGE("no last comma found in IDN string" ); 
	   hpe1445_LOG_STATUS( vi, thisPtr, hpe1445_INSTR_ERROR_UNEXPECTED); 
        }

        strcpy(instr_rev, last_comma+1);
	
	hpe1445_LOG_STATUS( vi, thisPtr, VI_SUCCESS); 
}
/* REGB_DRIVER _rev_query */
#endif

/****************************************************************************
hpe1445_timeOut
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1445_init()
   ------------------------------------------------------------------------
  | timeOut                                           ViInt32     IN
  |   ---------------------------------------------------------------------
  |  | This value sets the I/O timeout for all functions in
  |  | the driver. It is specified in milliseconds.

*****************************************************************************/
/* ----------------------------------------------------------------------- */
/* Purpose:  Changes the timeout value of the instrument.  Input is in     */
/*           milliseconds.                                                 */
/* ----------------------------------------------------------------------- */
ViStatus _VI_FUNC hpe1445_timeOut (
#ifdef MSGB_DRIVER
ViSession vi,
#endif
#ifdef REGB_DRIVER
INST_PARAM(vi, "", REGISTER), 
#endif
ViInt32 timeOut)
#ifdef MSGB_DRIVER
{
   ViStatus errStatus = 0;
   struct hpe1445_globals *thisPtr;
   

   GetGlobals(vi,thisPtr)
   hpe1445_DEBUG_CHK_THIS( vi, thisPtr );

   hpe1445_CHK_LONG_RANGE( timeOut, 1, 2147483647, VI_ERROR_PARAMETER2 );

   errStatus = viSetAttribute(vi, VI_ATTR_TMO_VALUE, timeOut);
   if ( VI_SUCCESS > errStatus)
   {
      hpe1445_LOG_STATUS( vi, thisPtr, errStatus );
   }

   hpe1445_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}
#endif

#ifdef REGB_DRIVER
{
   ViStatus errStatus = 0;
   struct hpe1445_globals *thisPtr;
   

   GetGlobals(vi,thisPtr)
   hpe1445_DEBUG_CHK_THIS( vi, thisPtr );

   hpe1445_CHK_LONG_RANGE( timeOut, 1, 2147483647, VI_ERROR_PARAMETER2 );

   if (thisPtr->e1406)
   {
	errStatus = viSetAttribute(vi, VI_ATTR_TMO_VALUE, timeOut);
   }
   else
   {
	errStatus = instr_settimeout(vi, timeOut);
   }
   if ( VI_SUCCESS > errStatus)
   {
      hpe1445_LOG_STATUS( vi, thisPtr, errStatus );
   }

   hpe1445_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}
#endif

/****************************************************************************
hpe1445_timeOut_Q
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1445_init()
   ------------------------------------------------------------------------
  | pTimeOut                                          ViPInt32    OUT
  |   ---------------------------------------------------------------------
  |  | This is the minimum timeout period that the driver
  |  | can be set to. It is specified in milliseconds.

*****************************************************************************/
/* ----------------------------------------------------------------------- */
/* Purpose:  Returns the current setting of the timeout value of the       */
/*           instrument in milliseconds.                                   */
/* ----------------------------------------------------------------------- */
ViStatus _VI_FUNC hpe1445_timeOut_Q (
#ifdef MSGB_DRIVER
ViSession vi,
#endif
#ifdef REGB_DRIVER
INST_PARAM(vi, "", REGISTER),
#endif
ViPInt32 timeOut)

#ifdef MSGB_DRIVER
{
   ViStatus errStatus = 0;
   struct hpe1445_globals *thisPtr;

   GetGlobals(vi,thisPtr)
   hpe1445_DEBUG_CHK_THIS( vi, thisPtr );

   errStatus = viGetAttribute(vi, VI_ATTR_TMO_VALUE, timeOut );
   if( VI_SUCCESS > errStatus)
   {
      hpe1445_LOG_STATUS( vi, thisPtr, errStatus );
   }

   hpe1445_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}
#endif

#ifdef REGB_DRIVER
{
   ViStatus errStatus = 0;
   struct hpe1445_globals *thisPtr;

   GetGlobals(vi,thisPtr)
   hpe1445_DEBUG_CHK_THIS( vi, thisPtr );

   if (thisPtr->e1406)
   {
	errStatus = viGetAttribute(vi, VI_ATTR_TMO_VALUE, timeOut );
   }
   else
   {
	errStatus = instr_gettimeout(vi, timeOut );
   }
   if( VI_SUCCESS > errStatus)
   {
      hpe1445_LOG_STATUS( vi, thisPtr, errStatus );
   }

   hpe1445_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}
#endif

/****************************************************************************
hpe1445_errorQueryDetect
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1445_init()
   ------------------------------------------------------------------------
  | errorQueryDetect                                  ViBoolean   IN
  |   ---------------------------------------------------------------------
  |  | Boolean which enables (VI_TRUE) or disables (VI_FALSE)
  |  | automatic instrument error querying.

*****************************************************************************/
ViStatus _VI_FUNC hpe1445_errorQueryDetect( 
#ifdef MSGB_DRIVER
ViSession vi,
#endif
#ifdef REGB_DRIVER
INST_PARAM(vi, "", REGISTER), 
#endif
  ViBoolean errDetect)
/*same for both driver types */
{
   ViStatus errStatus = 0;
   struct hpe1445_globals *thisPtr;

   GetGlobals(vi,thisPtr)
   hpe1445_DEBUG_CHK_THIS( vi, thisPtr );

   hpe1445_CHK_BOOLEAN( errDetect, VI_ERROR_PARAMETER2 );

   thisPtr->errQueryDetect = errDetect;

   hpe1445_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/****************************************************************************
hpe1445_errorQueryDetect_Q
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1445_init()
   ------------------------------------------------------------------------
  | pErrDetect                                        ViPBoolean  OUT
  |   ---------------------------------------------------------------------
  |  | Boolean indicating if automatic instrument error
  |  | querying is performed.

*****************************************************************************/
ViStatus _VI_FUNC hpe1445_errorQueryDetect_Q( 
#ifdef MSGB_DRIVER
ViSession vi,
#endif
#ifdef REGB_DRIVER
  INST_PARAM(vi, "", REGISTER), 
#endif
  ViPBoolean pErrDetect)
/* same for both types of driver */
{
   ViStatus errStatus = 0;
   struct hpe1445_globals *thisPtr;

   GetGlobals(vi,thisPtr)
   hpe1445_DEBUG_CHK_THIS( vi, thisPtr );

   *pErrDetect = thisPtr->errQueryDetect;

   hpe1445_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/****************************************************************************
hpe1445_dcl
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1445_init()

*****************************************************************************/
ViStatus _VI_FUNC hpe1445_dcl( 
#ifdef MSGB_DRIVER
ViSession vi
#endif
#ifdef REGB_DRIVER
  INST_PARAM(vi, "", REGISTER)
#endif
)

#ifdef MSGB_DRIVER

{
   ViStatus errStatus = 0;
   struct hpe1445_globals *thisPtr;

   GetGlobals(vi,thisPtr)
   hpe1445_DEBUG_CHK_THIS( vi, thisPtr );

   errStatus = viClear(vi);
   if( VI_SUCCESS > errStatus) { hpe1445_LOG_STATUS( vi, thisPtr, errStatus ); }	

   hpe1445_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}
#endif

#ifdef REGB_DRIVER
{
   ViStatus errStatus = 0;
   struct hpe1445_globals *thisPtr;

   GetGlobals(vi,thisPtr)
   hpe1445_DEBUG_CHK_THIS( vi, thisPtr );

   if (thisPtr->e1406)
   {
	errStatus = viClear(vi);
	if( VI_SUCCESS > errStatus) { hpe1445_LOG_STATUS( vi, thisPtr, errStatus ); }	
   }
   else
   {
   	INST_CLEAR(vi);
   }

   hpe1445_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}
#endif


/* Begin status functions */
/****************************************************************************
hpe1445_readStatusByte_Q
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1445_init()
   ------------------------------------------------------------------------
  | statusByte                                        ViPInt16    OUT
  |   ---------------------------------------------------------------------
  |  | returns the contents of the status byte

*****************************************************************************/
ViStatus _VI_FUNC hpe1445_readStatusByte_Q(
#ifdef MSGB_DRIVER
ViSession vi,
#endif
#ifdef REGB_DRIVER
INST_PARAM(vi, "", REGISTER),
#endif
ViPInt16 statusByte)

#ifdef MSGB_DRIVER

{
   ViStatus errStatus = 0;
   struct hpe1445_globals *thisPtr;
   ViUInt16 stb;

   GetGlobals(vi,thisPtr);
   hpe1445_DEBUG_CHK_THIS( vi, thisPtr );

   errStatus = viReadSTB(vi,&stb);
   if( VI_SUCCESS > errStatus)
	{
		hpe1445_LOG_STATUS(vi, thisPtr, errStatus );
	}

   *statusByte = (ViInt16)stb;

   hpe1445_LOG_STATUS(vi, thisPtr, VI_SUCCESS );
}
#endif

#ifdef REGB_DRIVER
{
   ViStatus errStatus = 0;
   struct hpe1445_globals *thisPtr;
   ViUInt16 stb;

   GetGlobals(vi,thisPtr);
   hpe1445_DEBUG_CHK_THIS( vi, thisPtr );

   if (thisPtr->e1406)
   {
	errStatus = viReadSTB(vi,&stb);
	if( VI_SUCCESS > errStatus)
	{
		hpe1445_LOG_STATUS(vi, thisPtr, errStatus );
	}
   }
   else
   {
	INST_READSTB(vi,&stb);
   }

   *statusByte = (ViInt16)stb;

   hpe1445_LOG_STATUS(vi, thisPtr, VI_SUCCESS );
}
#endif

/* hpe1445_operEvent_Q */
ViStatus _VI_FUNC hpe1445_operEvent_Q(
#ifdef MSGB_DRIVER
ViSession vi,
#else
INST_PARAM(vi,"",REGISTER),
#endif
ViPInt32 val)
{
 ViStatus errStatus;
 struct hpe1445_globals *thisPtr;
 ViInt32 count;
 char buf[20];

 GetGlobals(vi,thisPtr)
#ifdef REGB_DRIVER
 if( thisPtr->e1406)
#endif
 {
  errStatus = viWrite(vi,"STAT:OPER:EVEN?",15,&count);
  if( errStatus < VI_SUCCESS) 
   	hpe1445_LOG_STATUS(vi, thisPtr, errStatus );

  errStatus = viRead(vi,buf,20,&count);
  doDelay( thisPtr->myDelay );

  if( errStatus < VI_SUCCESS) 
   	hpe1445_LOG_STATUS(vi, thisPtr, errStatus );
  buf[count]='\0';
  *val = atoi( buf);
 }
#ifdef REGB_DRIVER
else
 {
  INST_QUERY(vi,"STAT:OPER:EVEN?","%d%*t",val);
 }
#endif
 hpe1445_LOG_STATUS(vi, thisPtr, VI_SUCCESS );
}

/* hpe1445_operCond_Q */
ViStatus _VI_FUNC hpe1445_operCond_Q(
#ifdef MSGB_DRIVER
ViSession vi,
#else
INST_PARAM(vi,"",REGISTER),
#endif
ViPInt32 val)
{
 ViStatus errStatus;
 struct hpe1445_globals *thisPtr;
 ViInt32 count;
 char buf[20];

 GetGlobals(vi,thisPtr)
#ifdef REGB_DRIVER
 if( thisPtr->e1406)
#endif
 {
  errStatus = viWrite(vi,"STAT:OPER:COND?",15,&count);
  if( errStatus < VI_SUCCESS) 
   	hpe1445_LOG_STATUS(vi, thisPtr, errStatus );

  errStatus = viRead(vi,buf,20,&count);
  doDelay( thisPtr->myDelay );

  if( errStatus < VI_SUCCESS) 
   	hpe1445_LOG_STATUS(vi, thisPtr, errStatus );
  buf[count]='\0';
  *val = atoi( buf);
 }
#ifdef REGB_DRIVER
else
 {
  INST_QUERY(vi,"STAT:OPER:COND?","%d%*t",val);
 }
#endif
 hpe1445_LOG_STATUS(vi, thisPtr, VI_SUCCESS );
}

/* hpe1445_statusQuesEven_Q */
ViStatus _VI_FUNC hpe1445_quesEvent_Q(
#ifdef MSGB_DRIVER
ViSession vi,
#else
INST_PARAM(vi,"",REGISTER),
#endif
ViPInt32 val)
{
 ViStatus errStatus;
 struct hpe1445_globals *thisPtr;
 ViInt32 count;
 char buf[20];

 GetGlobals(vi,thisPtr)
#ifdef REGB_DRIVER
 if( thisPtr->e1406)
#endif
 {
  errStatus = viWrite(vi,"STAT:QUES:EVEN?",15,&count);
  if( errStatus < VI_SUCCESS) 
   	hpe1445_LOG_STATUS(vi, thisPtr, errStatus );

  errStatus = viRead(vi,buf,20,&count);
  doDelay( thisPtr->myDelay );

  if( errStatus < VI_SUCCESS) 
   	hpe1445_LOG_STATUS(vi, thisPtr, errStatus );
  buf[count]='\0';
  *val = atoi( buf);
 }
#ifdef REGB_DRIVER
else
 {
  INST_QUERY(vi,"STAT:QUES:EVEN?","%d%*t",val);
 }
#endif
 hpe1445_LOG_STATUS(vi, thisPtr, VI_SUCCESS );
}

/* hpe1445_QuesCond_Q */
ViStatus _VI_FUNC hpe1445_quesCond_Q(
#ifdef MSGB_DRIVER
ViSession vi,
#else
INST_PARAM(vi,"",REGISTER),
#endif
ViPInt32 val)
{
 ViStatus errStatus;
 struct hpe1445_globals *thisPtr;
 ViInt32 count;
 char buf[20];

 GetGlobals(vi,thisPtr)
#ifdef REGB_DRIVER
 if( thisPtr->e1406)
#endif
 {
  errStatus = viWrite(vi,"STAT:QUES:COND?",15,&count);
  if( errStatus < VI_SUCCESS) 
   	hpe1445_LOG_STATUS(vi, thisPtr, errStatus );

  errStatus = viRead(vi,buf,20,&count);
  doDelay( thisPtr->myDelay );

  if( errStatus < VI_SUCCESS) 
   	hpe1445_LOG_STATUS(vi, thisPtr, errStatus );
  buf[count]='\0';
  *val = atoi( buf);
 }
#ifdef REGB_DRIVER
else
 {
  INST_QUERY(vi,"STAT:QUES:COND?","%d%*t",val);
 }
#endif
 hpe1445_LOG_STATUS(vi, thisPtr, VI_SUCCESS );
}



/* The scpi escape funnctions */

/* 	hpe1445_cmdString_Q
	Scpi escape function that returns a string.
	
	input:
		vi			- session
		p1			- The scpi command to be exectuted
		p2			- The size of p3
	
	output:
		p3			- the string returned by the instrument
		
*/
ViStatus _VI_FUNC hpe1445_cmdString_Q(
#ifdef MSGB_DRIVER 
ViSession vi,
#else
INST_PARAM(vi, "", REGISTER),
#endif
ViString p1,ViInt32 p2,ViChar _VI_FAR p3[] )
{
   ViStatus errStatus;
   ViInt32 mySize;
   struct hpe1445_globals *thisPtr;
   

   GetGlobals(vi,thisPtr);
/* hopefully this can be expanded on WIN32 */
   if( p2 < 2 || p2 > 32767)
      		hpe1445_LOG_STATUS(vi,thisPtr,VI_ERROR_PARAMETER2);
	
   /* If (message-based I/O) then write to instrument. */
#ifdef REGB_DRIVER
   if (thisPtr->e1406  )
#endif
   {
#ifdef NO_FORMAT_IO
      ViUInt32 rtn_size;
      if( (errStatus = viWrite(vi, p1, strlen(p1), &rtn_size )) < VI_SUCCESS)
      		hpe1445_LOG_STATUS(vi,thisPtr,errStatus);
#else
      if( (errStatus = viPrintf(vi, "%s\n",p1 )) < VI_SUCCESS)
      		hpe1445_LOG_STATUS(vi,thisPtr,errStatus);
#endif
      
      mySize = p2-1;
      
#ifdef NO_FORMAT_IO
      if( (errStatus = viRead(vi,  p3,mySize,&rtn_size)) < VI_SUCCESS)
      		hpe1445_LOG_STATUS(vi,thisPtr,errStatus);
      p3[rtn_size] = '\0';
      if( errStatus == VI_SUCCESS_MAX_CNT)
	{
		char * x = (char *)malloc(4096);
		if( !x )  hpe1445_LOG_STATUS(vi,thisPtr,VI_ERROR_ALLOC);
		while(1)
			{
      			 if( (errStatus = viRead(vi,  x,4096,&rtn_size)) 
				< VI_SUCCESS) 
				   hpe1445_LOG_STATUS(vi,thisPtr,errStatus);

			 if( errStatus == VI_SUCCESS_MAX_CNT) continue;
			 free( x );
			 break;
			}
	}
#else
      {
      int sz;
      sz = (int)p2;
	  if( (errStatus = viScanf(vi, "%#t", &sz,p3)) < VI_SUCCESS)
      		hpe1445_LOG_STATUS(vi,thisPtr,errStatus);
	  doDelay( thisPtr->myDelay );

      p3[sz]='\0';
      }
#endif
      
   }
#ifdef REGB_DRIVER
   else  /* Must be register-based I/O */
   {         
      ViInt32 i;
      mySize = p2;
      cscpi_exe(vi, p1,strlen(p1), p3, mySize);
      for(i=0; i<mySize; i++) if( p3[i] == '\n') break;
      
      p3[i]='\0';
      
   }
#endif

   hpe1445_LOG_STATUS( vi, thisPtr, errStatus );

} 

/*	PREFXI_cmd:
	Send a scpi command, it does not look for a response
	
	input:
		vi			- session
		p1			- Scpi command string

	comment:
		assumes p1 is NULL terminated C string.
*/

ViStatus _VI_FUNC hpe1445_cmd( 
#ifdef MSGB_DRIVER 
ViSession vi,
#else
INST_PARAM(vi, "", REGISTER),
#endif
ViString p1)

{
   ViInt32  bogus_size=50;             /* Parameter required for cscpi_exe() */
#ifdef NO_FORMAT_IO
   ViUInt32 rtn_size;                  /* Require parameter for viWrite().   */
#endif
   ViStatus errStatus;
   struct   hpe1445_globals *thisPtr;

   
   GetGlobals(vi,thisPtr)
   /* If (message-based I/O) then write to instrument. */
#ifdef REGB_DRIVER
   if (thisPtr->e1406)
#endif
   {
#ifdef NO_FORMAT_IO
      errStatus = viWrite(vi, p1, strlen(p1), &rtn_size );
#else 
      errStatus = viPrintf(vi,"%s\n", p1);
#endif
   }
#ifdef REGB_DRIVER
   else  /* Must be register-based I/O */
   {  
      ViChar   bogus_array[50];           /* Parameter required for cscpi_exe() */
      cscpi_exe(vi, p1, strlen(p1), bogus_array, bogus_size);
   }
#endif
   
   hpe1445_LOG_STATUS( vi, thisPtr, errStatus );
}


/*	PREFXI_cmdString:
	Send a scpi command, and then a string. It does not look for a response
	
	input:
		vi			- session
		p1			- Scpi command string
        p2          - Parameter in the form of a string.
        
	comment:
		assumes p1 & p2 are NULL terminated C strings.
*/
ViStatus _VI_FUNC hpe1445_cmdString( 
#ifdef MSGB_DRIVER 
ViSession vi,
#else
INST_PARAM(vi, "", REGISTER),
#endif
ViString p1,
ViString p2,
ViBoolean p3)
{  
   ViInt32  bogus_size=50;             /* Parameter required for cscpi_exe() */
#ifdef NO_FORMAT_IO
   ViUInt32 rtn_size;                  /* Require parameter for viWrite().   */
#endif
   ViStatus errStatus;
   struct   hpe1445_globals *thisPtr;

   GetGlobals(vi,thisPtr)
   /* If (message-based I/O) then write to instrument. */
#ifdef REGB_DRIVER
   if (thisPtr->e1406)
#endif
   {
#ifdef NO_FORMAT_IO 
   {
      ViChar   catString[512];  
      strcat(catString,p1," ",p2);
      errStatus = viWrite(vi,catString, strlen(catString), &rtn_size ); // fix this
   }
#else
      if(p3==VI_TRUE)
      {
       errStatus = viPrintf(vi,"%s %s\n", p1,p2);
      }
      else
      {
       errStatus = viPrintf(vi,"%s %s;:", p1,p2);
      }
#endif
   }
#ifdef REGB_DRIVER
   else  /* Must be register-based I/O */
   {  
      ViChar   catString[512];
      ViChar   bogus_array[50];           /* Parameter required for cscpi_exe() */
      strcat(catString,p1,p2);
      cscpi_exe(vi, catString, strlen(catString), bogus_array, bogus_size);
   }
#endif
   
   hpe1445_LOG_STATUS( vi, thisPtr, errStatus );
}


 
/*	hpe1445_cmdInt32_Q:
	Sends scpi command and waits for a response that must be representable as an int32
	
	input:
		vi			- session
		p1			- NULL terminated C string with scpi command.
	output:
		p2			- Result
		
	comment:
		a non number instrument response would yeild zero in p2.
*/	
ViStatus _VI_FUNC hpe1445_cmdInt32_Q(
#ifdef MSGB_DRIVER 
ViSession vi,
#else
INST_PARAM(vi, "", REGISTER),
#endif
ViString p1, ViPInt32 p2)
{  
#ifdef NO_FORMAT_IO
   ViUInt32 rtn_size;
#endif
   ViStatus errStatus;
   struct hpe1445_globals *thisPtr;
   

   GetGlobals(vi,thisPtr)
   /* If (message-based I/O) then write to instrument. */
#ifdef REGB_DRIVER
   if (thisPtr->e1406)
#endif
   {
#ifdef NO_FORMAT_IO
      char lc[100];
      if( (errStatus = viWrite(vi, p1, strlen(p1), &rtn_size )) < VI_SUCCESS)
      		hpe1445_LOG_STATUS(vi,thisPtr,errStatus);

      if( (errStatus = viRead(vi,lc,100,&rtn_size)) < VI_SUCCESS)
      		hpe1445_LOG_STATUS(vi,thisPtr,errStatus);
      *p2 = atol(lc);
#else
      if( (errStatus = viPrintf(vi, "%s\n",p1)) < VI_SUCCESS)
      		hpe1445_LOG_STATUS(vi,thisPtr,errStatus);

      if( (errStatus = viScanf(vi,"%ld%*t",p2)) < VI_SUCCESS)
      		hpe1445_LOG_STATUS(vi,thisPtr,errStatus);
	  doDelay( thisPtr->myDelay );

#endif	
        
   }
#ifdef REGB_DRIVER
   else  /* Must be register-based I/O */
   {               
      char lc[100];
      cscpi_exe(vi, p1,strlen(p1), lc, 100);
      
      *p2 = atol( lc );
   }
#endif

   hpe1445_LOG_STATUS( vi, thisPtr, errStatus );

} 

/*	hpe1445_cmdReal64_Q
	sends scpi command that must elicit a response that can be represented as a real64 (double)
	
	input:
		vi			- session
		p1			- NULL terminated C string containing scpi command
	
	output:
		p2			- Response converted to real 64
		
	comment:
		non numeric response will yeild 0 in p2, in case of underflow underfined.
*/
	
ViStatus _VI_FUNC hpe1445_cmdReal64_Q(
#ifdef MSGB_DRIVER 
ViSession vi,
#else
INST_PARAM(vi, "", REGISTER),
#endif
ViString p1, ViPReal64 p2)
{   
#ifdef NO_FORMAT_IO
   ViUInt32 rtn_size;
#endif
   ViStatus errStatus;
   struct hpe1445_globals *thisPtr;
   


   GetGlobals(vi,thisPtr)
   /* If (message-based I/O) then write to instrument. */
#ifdef REGB_DRIVER
   if (thisPtr->e1406)
#endif
   {
#ifdef NO_FORMAT_IO
	  char lc[100];
      if( (errStatus = viWrite(vi, p1, strlen(p1), &rtn_size )) < VI_SUCCESS)
      		hpe1445_LOG_STATUS(vi,thisPtr,errStatus);

      if( (errStatus = viRead(vi,lc,100,&rtn_size)) < VI_SUCCESS)
      		hpe1445_LOG_STATUS(vi,thisPtr,errStatus);
      *p2 = atof(lc);
#else
      if( (errStatus = viPrintf(vi, "%s\n",p1)) < VI_SUCCESS)
      		hpe1445_LOG_STATUS(vi,thisPtr,errStatus);

      if( (errStatus = viScanf(vi,"%lf%*t",p2)) < VI_SUCCESS)
      		hpe1445_LOG_STATUS(vi,thisPtr,errStatus);
	  doDelay( thisPtr->myDelay );

#endif	
        
   }
#ifdef REGB_DRIVER
   else  /* Must be register-based I/O */
   {            
      char lc[100];
      cscpi_exe(vi, p1,strlen(p1), lc, 100);
      
      *p2 = atof( lc );
   }
#endif

   hpe1445_LOG_STATUS( vi, thisPtr, errStatus );

} 

/*	hpe1445_cmdInt16_Q:
	Scpi command that must respond with a number that can be interpreted as an int16
	
	input:
		vi			- session
		p1			- NULL terminated C string with command
	
	output:
		p2			- response converted to int 16
		
	comment:
		will return 0 if response is not a valid number
*/
 
ViStatus _VI_FUNC hpe1445_cmdInt16_Q(
#ifdef MSGB_DRIVER 
ViSession vi,
#else
INST_PARAM(vi, "", REGISTER),
#endif
ViString p1, ViPInt16 p2)
{
   ViStatus errStatus;
   struct hpe1445_globals *thisPtr;
   
   GetGlobals(vi,thisPtr)
   /* If (message-based I/O) then write to instrument. */
#ifdef REGB_DRIVER
   if (thisPtr->e1406)
#endif
   {
#ifdef NO_FORMAT_IO
	  char lc[100];
	  ViUInt32 rtn_size;
      if( (errStatus = viWrite(vi, p1, strlen(p1), &rtn_size )) < VI_SUCCESS)
      		hpe1445_LOG_STATUS(vi,thisPtr,errStatus);

      if( (errStatus = viRead(vi,lc,100,&rtn_size)) < VI_SUCCESS)
      		hpe1445_LOG_STATUS(vi,thisPtr,errStatus);
      *p2 = atoi(lc);
#else
      if( (errStatus = viPrintf(vi, "%s\n",p1)) < VI_SUCCESS)
      		hpe1445_LOG_STATUS(vi,thisPtr,errStatus);

      if( (errStatus = viScanf(vi,"%hd%*t",p2)) < VI_SUCCESS)
      		hpe1445_LOG_STATUS(vi,thisPtr,errStatus);
	  doDelay( thisPtr->myDelay );

#endif	
        
   }
#ifdef REGB_DRIVER
   else  /* Must be register-based I/O */
   {  
      char lc[100];       
      cscpi_exe(vi, p1,strlen(p1), lc, 100);
      
      *p2 = atoi( lc );
   }
#endif

   hpe1445_LOG_STATUS( vi, thisPtr, errStatus );

} 

/* 	hpe1445_cmdInt16Arr_Q
	Scpi command that returns a int16 definate arbitrary block
	
	input:
		vi			- session
		p1			- NULL terminated C string containing scpi command
		p2			- size of p3
	
	output:
		p3			- reutrns array of int16s
		p4			- # of items in p3
	
*/
ViStatus _VI_FUNC hpe1445_cmdInt16Arr_Q(
#ifdef MSGB_DRIVER 
ViSession vi,
#else
INST_PARAM(vi, "", REGISTER),
#endif
ViString p1,ViInt32 p2,ViInt16 _VI_FAR p3[],ViPInt32 p4)
/*	This entry point assumes that the return result is a arbitrary block.  Do not use
	this entry point for ascii values.  For that just get the data from _cmd_string_Q
	routine.
*/
{
	ViStatus errStatus ;
	ViUInt32 cnt;
	struct hpe1445_globals * thisPtr;
	char lc[100];
	ViInt32 i, ArrSize,theCount,nbytes;
	char _huge *Array;
	
   GetGlobals(vi,thisPtr)

   if( p2 < hpe1445_CMDINT16ARR_Q_MIN || p2 > hpe1445_CMDINT16ARR_Q_MAX)
			hpe1445_LOG_STATUS(vi,thisPtr,VI_ERROR_PARAMETER3);

   /* If (message-based I/O) then write to instrument. */
#ifdef REGB_DRIVER
   if (thisPtr->e1406)
#endif
	{	
		if( (errStatus = viWrite(vi,p1,strlen(p1),&cnt)) < VI_SUCCESS)
			hpe1445_LOG_STATUS(vi,thisPtr,errStatus);
			
		if( (errStatus = viRead(vi,lc,1,&cnt)) < VI_SUCCESS)
			hpe1445_LOG_STATUS(vi,thisPtr,errStatus);
	
		if( lc[0] != '#')
		{
		  	hpe1445_LOG_STATUS(vi,thisPtr,VI_ERROR_INV_EXPR);
		}
		
		if( (errStatus = viRead(vi,lc,1,&cnt)) < VI_SUCCESS)
			hpe1445_LOG_STATUS(vi,thisPtr,errStatus);
	
		nbytes = lc[0]-'0';
	
		if( (errStatus = viRead(vi,lc,nbytes,&cnt)) < VI_SUCCESS)
			hpe1445_LOG_STATUS(vi,thisPtr,errStatus);
			
		ArrSize = atol(lc);
#ifdef WIN32
		Array = (char *)malloc(ArrSize*sizeof(ViInt16) ); 
#else		                             
		Array = (char _huge *)halloc(ArrSize, 2 ); /* float 32 better be 4 bytes!*/
#endif	
		if( !Array ) hpe1445_LOG_STATUS(vi,thisPtr,VI_ERROR_ALLOC);
	
		if( (errStatus = viRead(vi,Array,ArrSize*sizeof(ViInt16),&cnt)) < VI_SUCCESS)
			hpe1445_LOG_STATUS(vi,thisPtr,errStatus);
	}   
#ifdef REGB_DRIVER
	else
	{
	  	ArrSize = p2; /* for cscpi exe the user MUST pass in a good length*/
	  	
	  	ArrSize += 4; /* DAB header + tail */
#ifdef WIN32
		Array = (char *)malloc(ArrSize*sizeof(ViInt16) ); 
#else	  	
	  	Array = (char _huge *)halloc(ArrSize, 2 ); 
#endif	
		if( !Array ) hpe1445_LOG_STATUS(vi,thisPtr,VI_ERROR_ALLOC);
	
		cscpi_exe( vi, p1, strlen(p1),Array, ArrSize*sizeof(ViInt16) );
		
		if( Array[0] != '#')
		{
		  	hpe1445_LOG_STATUS(vi,thisPtr,VI_ERROR_INV_EXPR);
		}
		
		nbytes = Array[1]-'0';
		
		strncpy(lc,Array+2,(int)nbytes);
		lc[nbytes]=0;
		
		ArrSize = atol(lc)/sizeof(ViInt16);
		
		Array += nbytes+2;
	}
#endif
	
	theCount = (p2*sizeof(ViInt16) < ArrSize*sizeof(ViInt16)) ? p2 * sizeof(ViInt16) :
			 ArrSize*sizeof(ViInt16);
	
	_fmemcpy(p3,Array,(size_t)theCount); /*copy only as many as the user allows*/
	
	*p4=theCount/sizeof(ViInt16);

     for(i=0; i<*p4; i++)
     	SWAP_16(&p3[i]);
	 
	 hpe1445_LOG_STATUS(vi,thisPtr,VI_SUCCESS);    	    	
}                                                                         

/* 	hpe1445_cmdInt
	Scpi command that takes ONE int16 or int32 command
	
	input:
		vi			- session
		p1			- NULL terminated C string containing scpi command
		p2			- integer parameter
	
*/
ViStatus _VI_FUNC hpe1445_cmdInt(
#ifdef MSGB_DRIVER 
ViSession vi,
#else
INST_PARAM(vi, "", REGISTER),
#endif
ViString p1,ViInt32 p2,ViBoolean p3 )
{  
   ViStatus errStatus;
   struct hpe1445_globals *thisPtr;

   GetGlobals(vi,thisPtr);
   /* If (message-based I/O) then write to instrument. */
#ifdef REGB_DRIVER
   if (thisPtr->e1406  )
#endif
   {
#ifdef NO_FORMAT_IO
      ViUInt32 rtn_size;
      char lc[300];
      wsprintf(lc,"%s %ld",p1,p2);
      if( (errStatus = viWrite(vi, lc, strlen(lc), &rtn_size )) < VI_SUCCESS)
      		hpe1445_LOG_STATUS(vi,thisPtr,errStatus);
#else
      if(p3 == VI_TRUE)
      {
       if( (errStatus = viPrintf(vi, "%s %ld\n",p1,p2 )) < VI_SUCCESS)
      		hpe1445_LOG_STATUS(vi,thisPtr,errStatus);  
      }
      else
      {
      if( (errStatus = viPrintf(vi, "%s %ld;:",p1,p2 )) < VI_SUCCESS)
      		hpe1445_LOG_STATUS(vi,thisPtr,errStatus);  
      }
#endif
   }

#ifdef REGB_DRIVER
   else  /* Must be register-based I/O */
   {         
      ViChar bogus_array[50];
      ViInt32  bogus_size=50;             /* Parameter required for cscpi_exe() */
      char lc[300];
      wsprintf(lc,"%s %ld",p1,p2);
      cscpi_exe(vi, lc,strlen(lc), bogus_array, bogus_size);
   }
#endif

   hpe1445_LOG_STATUS( vi, thisPtr, errStatus );

} 

/* 	hpe1445_cmdReal
	Scpi command that takes ONE Real32 or Real64 param
	
	input:
		vi			- session
		p1			- NULL terminated C string containing scpi command
		p2			- Real parameter
	
*/
ViStatus _VI_FUNC hpe1445_cmdReal(
#ifdef MSGB_DRIVER 
ViSession vi,
#else
INST_PARAM(vi, "", REGISTER),
#endif
ViString p1,ViReal64 p2,ViBoolean p3 )
{

   ViStatus errStatus;
   struct hpe1445_globals *thisPtr;

   GetGlobals(vi,thisPtr);
   /* If (message-based I/O) then write to instrument. */
#ifdef REGB_DRIVER
   if (thisPtr->e1406  )
#endif
   {
#ifdef NO_FORMAT_IO
	  ViUInt32 rtn_size;
   	  char lc[300];
      int dec,sign;
      char sign_str[4];
      char * myNum;
      myNum = _fcvt(p2,10,&dec,&sign);
      if( sign ) strcpy(sign_str," -.");
      else strcpy(sign_str," .");
      wsprintf(lc,"%s%s.%sE%d",p1,sign_str,myNum,dec);
      if( (errStatus = viWrite(vi, lc, strlen(lc), &rtn_size )) < VI_SUCCESS)
      		hpe1445_LOG_STATUS(vi,thisPtr,errStatus);
#else
      if(p3==VI_TRUE)
      {
       if( (errStatus = viPrintf(vi, "%s %g\n",p1,p2 )) < VI_SUCCESS)
      		hpe1445_LOG_STATUS(vi,thisPtr,errStatus); 
      }
      else
      {
       if( (errStatus = viPrintf(vi, "%s %g;:",p1,p2 )) < VI_SUCCESS)
      		hpe1445_LOG_STATUS(vi,thisPtr,errStatus); 
      }
      		
#endif
   }

#ifdef REGB_DRIVER
   else  /* Must be register-based I/O */
   {         
      ViChar bogus_array[50];
      ViInt32  bogus_size=50;
      char * myNum;             
      int dec,sign;
      char sign_str[4];
      char lc[300];
      myNum = _fcvt(p2,10,&dec,&sign);
      if( sign ) strcpy(sign_str," -.");
      else strcpy(sign_str," .");
      wsprintf(lc,"%s%s.%sE%d",p1,sign_str,myNum,dec);
      cscpi_exe(vi, lc,strlen(lc), bogus_array, bogus_size);
   }
#endif

   hpe1445_LOG_STATUS( vi, thisPtr, errStatus );

} 


/* 	hpe1445_opc_Q
	Returns VI_TRUE if operations still pending
	
	input:
		vi			- session
	output
		p1			- VI_TRUE if operations pending
	
*/
ViStatus _VI_FUNC hpe1445_opc_Q(
#ifdef MSGB_DRIVER 
ViSession vi,
#else
INST_PARAM(vi, "", REGISTER),
#endif
ViPBoolean p1)
{
   ViInt32 tmp;
   ViStatus errStatus;
   struct hpe1445_globals *thisPtr;
   GetGlobals(vi,thisPtr);
   /* If (message-based I/O) then write to instrument. */
#ifdef REGB_DRIVER
   if (thisPtr->e1406  )
#endif
   {
#ifdef NO_FORMAT_IO
      ViInt32 count;
	  
      char lc[256];
      if( (errStatus = viWrite(vi, "*OPC?",5,&count )) < VI_SUCCESS)
      		hpe1445_LOG_STATUS(vi,thisPtr,errStatus);
      
      if( (errStatus = viRead(vi, lc,256,&count )) < VI_SUCCESS)
      		hpe1445_LOG_STATUS(vi,thisPtr,errStatus);
      
	lc[count] = '\0';
	*p1 = atoi( lc );

#else
      if( (errStatus = viPrintf(vi, "*OPC?" )) < VI_SUCCESS)
      		hpe1445_LOG_STATUS(vi,thisPtr,errStatus);
      if( (errStatus = viScanf(vi, "%d%*t",&tmp )) < VI_SUCCESS)
      		hpe1445_LOG_STATUS(vi,thisPtr,errStatus);
	  *p1=(ViBoolean)tmp;
	  doDelay( thisPtr->myDelay );

#endif
   }

#ifdef REGB_DRIVER
   else  /* Must be register-based I/O */
   {         
      INST_QUERY(vi,"*OPC?","%d",p1);
   }
#endif

   hpe1445_LOG_STATUS( vi, thisPtr, errStatus );

} 

/*
	AutoConnectToFirst finds the first device in the FindFirst, FindNext
	order.  Note at this time it WILL find the GPIB first in preference
	to the GPIB-VXI.
*/
ViStatus _VI_FUNC hpe1445_autoConnectToFirst( ViPSession vi )
{
	ViStatus errStatus;
	ViFindList fl;
	ViSession defRM;
	ViUInt32 Match;
	char res[100];
	int i;
	ViInt32 retcount;
	ViSession instr;
	ViUInt16 intfType;
	ViUInt16 manfId;
	ViUInt16 modelCode;
	char buf[256];

    *vi = VI_NULL;
	
	errStatus = viOpenDefaultRM( &defRM );
	if( errStatus < VI_SUCCESS) return errStatus;

	errStatus = viFindRsrc(defRM,"?*INSTR",&fl,&Match,res);
	if( errStatus < VI_SUCCESS ) 
		{
		  viClose(defRM);
		  return errStatus; 
       	}

    instr = 0;
	
	for(i=0; i<(int)Match; i++) //in this loop search for your device
	{
		 if( instr ) viClose(instr);	 
		 	
		 if( i ) viFindNext(fl,res);
		 if( viOpen(defRM,res,VI_NULL,VI_NULL,&instr) != VI_SUCCESS ) continue; 
		 if( viGetAttribute( instr, VI_ATTR_INTF_TYPE, &intfType) != VI_SUCCESS  ) continue;
		 
     	 switch( intfType)
		 {
			case VI_INTF_GPIB:

				errStatus = viWrite(instr,(ViBuf)"*IDN?\n",5,&retcount);
	            if( VI_SUCCESS > errStatus)continue;
                                
				errStatus = viRead(instr, (ViBuf)buf,100,&retcount);
                                {
				 DWORD t = GetTickCount();
                                 while(GetTickCount()-t < 10);
				}
				if( VI_SUCCESS > errStatus) continue;
                
				/* YOUR DEVICE GOES HERE*/
				if( !strstr(buf,hpe1445_IDN_STRING)) continue;                                          
                break; //found an instrument.

			case VI_INTF_VXI:
			case VI_INTF_GPIB_VXI: 

				/* find the VXI manfacturer's ID */
				if( viGetAttribute(instr, VI_ATTR_MANF_ID, &manfId) != VI_SUCCESS ) continue; /* just avoid UAE */
				if( viGetAttribute( instr, VI_ATTR_MODEL_CODE, (ViPAttrState)(&modelCode)) != VI_SUCCESS ) continue;
				if( (manfId != hpe1445_MANF_ID) ||(modelCode != hpe1445_MODEL_CODE) ) continue; /* not our instrument */
				break;
			
			default:
				continue;

		 }	/* switch - id_query */
		      
		 /* Coming here means that we found the instrument*/
		viClose(defRM); /* closes everything */
	    /* now call the _init routine */
		return hpe1445_init(res,VI_FALSE,VI_FALSE,vi);
	}  /* for */
	
	viClose(defRM);
	return VI_ERROR_RSRC_NFOUND;
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1445_arbWave
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This function clears the E1445 of all previously defined
 *           waveforms and defines a new one.  The segments created with
 *           createSeg are stored in computer memory and not in the
 *           instrument's waveform segment
 *            The arbitrary waveform is created from the segment and dwell
 *           list, and is then output.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument handle. In this case VI_NULL.
 * 
 * PARAM 2 : ViReal64 samplerate
 * IN        
 *            Sample rate of waveform.
 * 
 *      MAX = hpe1445_ARBRATEMAX   42.94e6
 *      MIN = hpe1445_ARBRATEMIN   0
 * 
 * PARAM 3 : ViReal64 ampl
 * IN        
 *            Positive full scale output amplitude (volts).
 * 
 *      MAX = hpe1445_ARBAMPL_MAX   12.05
 *      MIN = hpe1445_ARBAMPL_MIN   -12.05
 * 
 * PARAM 4 : ViString sequence
 * IN        
 *            The name given to the segment sequence, must be unique.
 * 
 * PARAM 5 : ViInt32 seqlength
 * IN        
 *            Number of segment names in the waveform.
 * 
 *      MAX = hpe1445_ARBSEQNO_MAX   32768
 *      MIN = hpe1445_ARBSEQNO_MIN   0
 * 
 * PARAM 6 : ViString seqorder
 * IN        
 *            A comma separated list of waveform segment names.  These
 *           segments must have (already) been created using the
 *           'hpe1445_createSeg' function. No spaces allowed in this list.
 * 
 * PARAM 7 : ViInt32 dwell[]
 * IN        
 *            The number of times to repeat a waveform segment in the
 *            segment list of the seqorder parameter. If VI_NULL,  each
 *           segment is output one time. Thus, dwell{4,2,2} will repeat the
 *           first segment four times, the second segment two times, and the
 *            third segment two times.
 * 
 * PARAM 8 : ViInt32 impedance
 * IN        
 *            Sets the output impedance to 50 or 75 ohms.
 * 
 *      Macro Name        Value  Description
 *      -----------------------------------------------------------
 *      hpe1445_Z_50         50  50 Ohm
 *      hpe1445_Z_75         75  75 Ohm
 * 
 * PARAM 9 : ViString load
 * IN        
 *            'Sets the output load. "MIN' (50), 'MAX' (75),  'INF'
 *           (infinity).
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           the "hpe1413_error_message" routine.
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1445_arbWave(
  ViSession vi,
  ViReal64 samplerate,
  ViReal64 ampl,
  ViString sequence,
  ViInt32 seqlength,
  ViString seqorder,
  ViInt32 dwell[],
  ViInt32 impedance,
  ViString load)
{
    ViStatus errStatus = 0;
    struct hpe1445_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ hpe1445_LOG_STATUS( vi, 0, errStatus ); }

    hpe1445_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1445_CDE_INIT( "hpe1445_arbWave" );

    /* Perform Error Checking on Each Parameter */
    hpe1445_CHK_REAL_RANGE(samplerate
                          ,hpe1445_ARBRATEMIN
                          ,hpe1445_ARBRATEMAX
                          ,VI_ERROR_PARAMETER2);

    hpe1445_CHK_REAL_RANGE(ampl
                          ,hpe1445_ARBAMPL_MIN
                          ,hpe1445_ARBAMPL_MAX
                          ,VI_ERROR_PARAMETER3);

    hpe1445_CHK_LONG_RANGE(seqlength
                          ,hpe1445_ARBSEQNO_MIN
                          ,hpe1445_ARBSEQNO_MAX
                          ,VI_ERROR_PARAMETER5);

    { 
      struct ListStruct *p;
      char *cp;
      int i;
      ViBoolean Markers = VI_FALSE;
    
      errStatus = viPrintf(vi,":ABORT\n");
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
      errStatus = viPrintf(vi,":VOLT:UNIT V\n;");
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
      errStatus = viPrintf(vi,":FUNC:USER NONE\n");
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
      errStatus = viPrintf(vi,":SOUR:LIST1:SSEQ:DEL:ALL\n");
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
      errStatus = viPrintf(vi,":SOUR:LIST1:SEGM:DEL:ALL\n");
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
      errStatus = viPrintf(vi,":SOUR:FREQ1:FIX %g;",samplerate);
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
      errStatus = viPrintf(vi,":SOUR:FUNC:SHAP USER;");
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
      if( impedance == 50 || impedance == 75)
      {
      errStatus = viPrintf(vi,":OUTP:IMP %d;",(int)impedance);
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
      }else {hpe1445_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER8 );}
     
    
      errStatus = viPrintf(vi,":OUTP:LOAD %s;",load);
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
      errStatus = viPrintf(vi,":SOUR:VOLT:LEV:IMM:AMPL %g\n",ampl);
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
      p=thisPtr->ListHead;
      while(p)
      {
    	p->loaded = VI_FALSE;
    	p=p->next;
      }
    
      cp = seqorder;
      while(1)
      {
        char *comma;
    	char name[25];
    
    	comma = strchr(cp,',');
    	if( !comma )
    	  strcpy(name,cp);
    	else
    	{
    	  strncpy(name,cp,comma-cp);
    	  name[comma-cp]=0;
    	}
    	p=thisPtr->ListHead;
    	while(p)
    		{
    			if( strcmp(name,p->name) == 0)
    			{
    			 if( !p->loaded ) 
    			 {
    			    int i;
    			    p->loaded = VI_TRUE;
    			    
    				errStatus = viPrintf(vi,":SOUR:LIST1:SEGM:SEL %s\n",name);
    			    if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    			 
    			    errStatus = viPrintf(vi,":SOUR:LIST1:SEGM:DEF %d\n",(int)p->size);
    			    if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
    				errStatus = viPrintf(vi,":SOUR:LIST1:SEGM:VOLT %g",p->data[0]);
    			    if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
    				for(i=1; i<p->size; i++)
    				{
    					errStatus = viPrintf(vi,",%g",p->data[i]);
    					if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    				}
    				errStatus = viPrintf(vi,"\n");
    			    if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
    				if( p->MarkerList )
    				{
    				  Markers = VI_TRUE;
    
    				  errStatus = viPrintf(vi,":SOUR:LIST1:SEGM:MARK %d",
    												(int)p->MarkerList[0]);
    				  if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
    				  for(i=1; i<p->size; i++)
    					{
    					  errStatus = viPrintf(vi,",%d",(int)p->MarkerList[i]);
    					  if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    					}
    				  
    				  errStatus = viPrintf(vi,"\n");
    			      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    				}
    			 }
    			}
    			p=p->next;
    			
    		}
        if( !comma ) break;
    	cp = comma+1;
      }
    
      errStatus = viPrintf(vi,":SOUR:LIST1:SSEQ:SEL %s\n",sequence);
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
      errStatus = viPrintf(vi,":SOUR:LIST1:SSEQ:DEF %d\n",(int)seqlength);
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
      errStatus = viPrintf(vi,":SOUR:LIST1:SSEQ:SEQ %s\n",seqorder);
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
      
      
      if( dwell )
      {
        errStatus = viPrintf(vi,":SOUR:LIST1:SSEQ:DWEL:COUN %d",(int)dwell[0]);
        if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    	for(i=1; i<seqlength;i++)
    	{
    		errStatus = viPrintf(vi,",%d",(int)dwell[i]);
    		if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
    	}
    	errStatus = viPrintf(vi,"\n");
    	if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
      }
      else
      {
        errStatus = viPrintf(vi,":SOUR:LIST1:SSEQ:DWEL:COUN 1\n");
        if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
      }
    
      errStatus = viPrintf(vi,":SOUR:FUNC:USER %s\n",sequence);
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
      if( Markers )
      {
    	errStatus = viPrintf(vi,":SOUR:MARK:STAT ON\n");
    	if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    	errStatus=viPrintf(vi,"SOUR:MARK:FEED \"SOUR:LIST1\"\n");
    	if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
      }
      else
      {
    	errStatus = viPrintf(vi,":SOUR:MARK:STAT OFF\n");
    	if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
      }
    
      errStatus = viPrintf(vi,"INIT:IMM\n");
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
     
    }
    hpe1445_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1445_couple1446
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This function is used to construct a coupled command consisting
 *            of the parameters set by the following SCPI commands:
 *           
 *            OUTPut2:ATTenuation
 *            OUTPut2:IMPedance
 *            SOURce2:VOLTage:LEVel:IMMediate:OFFSet
 *           
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument handle. In this case VI_NULL.
 * 
 * PARAM 2 : ViInt32 attenuation
 * IN        
 *            Agilent E1446 output attenuation (dB):  0, 20, or
 *           hpe1445_46UNUSED (=-20).
 * 
 *      Macro Name               Value  Description
 *      -----------------------------------------------------------
 *      hpe1445_46ATTENZERO          0  Zero
 *      hpe1445_46ATTEN20           20  Twenty
 *      hpe1445_46UNUSED           -20  Unused
 * 
 * PARAM 3 : ViInt32 impedance
 * IN        
 *            Agilent E1446 output impedance:  0, 50, 75 or hpe1445_46UNUSED
 *           (-20).
 * 
 *      Macro Name            Value  Description
 *      -----------------------------------------------------------
 *      hpe1445_46IMP0            0  Zero
 *      hpe1445_46IMP50          50  Fifty
 *      hpe1445_46IMP75          75  Seventy five
 *      hpe1445_46UNUSED        -20  Unused
 * 
 * PARAM 4 : ViReal64 offset
 * IN        
 *            Sets the Agilent E1446 offset voltage from -9.999695 to +10.0,
 *           or hpe1445_46UNUSED (-20).
 * 
 *      MAX = hpe1445_46OFFMAX   10.0
 *      MIN = hpe1445_46UNUSED   -20
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1413_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1445_couple1446(
  ViSession vi,
  ViInt32 attenuation,
  ViInt32 impedance,
  ViReal64 offset)
{
    ViStatus errStatus = 0;
    struct hpe1445_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ hpe1445_LOG_STATUS( vi, 0, errStatus ); }

    hpe1445_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1445_CDE_INIT( "hpe1445_couple1446" );

    /* Perform Error Checking on Each Parameter */
    hpe1445_CHK_REAL_RANGE(offset
                          ,hpe1445_46UNUSED
                          ,hpe1445_46OFFMAX
                          ,VI_ERROR_PARAMETER4);

    {
      char lc[512];
      char tb[512];
    
      
      tb[0]=0;
      if( !thisPtr->CoupleBuffer ) 
      {
        
    	return hpe1445_ERR_NOBEGIN ;
      }
      
      if( attenuation>=0 )
      {
       sprintf(lc,":OUTP2:ATT %d;",(int)attenuation);
       strcat(tb,lc);
      }else if( attenuation != hpe1445_46UNUSED ) return VI_ERROR_PARAMETER2;
    
      if( impedance==0 || impedance == 50 || impedance==75)
      {
       sprintf(lc,":OUTP2:IMP %d;",(int)impedance);
       strcat(tb,lc);
    
      }	else if( attenuation != hpe1445_46UNUSED) return VI_ERROR_PARAMETER3;
    
     
     if( offset >= -9.999695)
      {
        sprintf(lc,":SOUR2:VOLT:OFFSET %g;",offset);
    	strcat(tb,lc);
    
      }else if( offset != hpe1445_46UNUSED) return VI_ERROR_PARAMETER4;
    
      
      if( !strlen(tb) ) return VI_SUCCESS;
    
      if( strlen(tb) + strlen( thisPtr->CoupleBuffer) >= (size_t)thisPtr->CoupleSize) 
    	{ 
    	 thisPtr->CoupleBuffer = (char *) realloc( thisPtr->CoupleBuffer,	
    		thisPtr->CoupleSize+1024); 
    	 if( !thisPtr->CoupleBuffer ) {return VI_ERROR_ALLOC;} 
    	 thisPtr->CoupleSize += 1024;
    	} 
      	
      strcat(thisPtr->CoupleBuffer,tb);
      
      return VI_SUCCESS;
    }
    hpe1445_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1445_coupleBegin
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Starts a coupled command and clears the couple buffer.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument handle. In this case VI_NULL.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           the "hpe1413_error_message" routine.
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1445_coupleBegin(
  ViSession vi)
{
    ViStatus errStatus = 0;
    struct hpe1445_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ hpe1445_LOG_STATUS( vi, 0, errStatus ); }

    hpe1445_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1445_CDE_INIT( "hpe1445_coupleBegin" );

    /* Perform Error Checking on Each Parameter */
    {
      if( !thisPtr->CoupleBuffer ) 
      {
    	thisPtr->CoupleBuffer = (char *)malloc(1024);
    	if( !thisPtr->CoupleBuffer) hpe1445_LOG_STATUS( vi, thisPtr, VI_ERROR_ALLOC );
      	thisPtr->CoupleSize = 1024;
      }
      thisPtr->CoupleBuffer[0]=0;  
    }
    hpe1445_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1445_coupleCGFV
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This function is used to construct a coupled command consisting
 *            of the parameters set by the following SCPI commands:
 *           
 *            SOURce:ARBitrary:DAC:SOURce
 *            SOURce:FUNCtion:SHAPe
 *            SOURceRAMP:POINts
 *           
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument handle. In this case VI_NULL.
 * 
 * PARAM 2 : ViString dacsour
 * IN        
 *            Selects the DAC data source: "DPOR", "INT", "LBUS", "VXI", or
 *           VI_NULL (unused).
 * 
 * PARAM 3 : ViString shape
 * IN        
 *            Selects the waveform shape:  "DC", "RAMP", "SIN", "SQU", "TRI",
 *           "USER", or VI_NULL (unused).
 * 
 * PARAM 4 : ViReal64 points
 * IN        
 *            Number of points to be used to generate the stepped ramp, or
 *            hpe1445_CGFVUNUSED (=-1) if unused.
 * 
 *      MAX = hpe1445_CGFVPOINTSMAX   262144
 *      MIN = hpe1445_CGFVUNUSED   -1
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           the "hpe1413_error_message" routine.
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1445_coupleCGFV(
  ViSession vi,
  ViString dacsour,
  ViString shape,
  ViReal64 points)
{
    ViStatus errStatus = 0;
    struct hpe1445_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ hpe1445_LOG_STATUS( vi, 0, errStatus ); }

    hpe1445_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1445_CDE_INIT( "hpe1445_coupleCGFV" );

    /* Perform Error Checking on Each Parameter */
    hpe1445_CHK_REAL_RANGE(points
                          ,hpe1445_CGFVUNUSED
                          ,hpe1445_CGFVPOINTSMAX
                          ,VI_ERROR_PARAMETER4);

    {
      char lc[512];
      char tb[512];
    
      
      tb[0]=0;
      if( !thisPtr->CoupleBuffer ) 
      {
        
    	return hpe1445_ERR_NOBEGIN ;
      }
      
      if( dacsour )
      {
       sprintf(lc,":ARB:DAC:SOUR %s;",dacsour);
       strcat(tb,lc);
      }
    
      if( shape)
      {
       sprintf(lc,":FUNC %s;",shape);
       strcat(tb,lc);
    
      }
     
     if( points >= 4)
      {
        sprintf(lc,":RAMP:POINTS %g;",points);
    	strcat(tb,lc);
    
      }else if( points != hpe1445_CGFVUNUSED) return VI_ERROR_PARAMETER4;
    
      
      if( !strlen(tb) ) return VI_SUCCESS;
    
      if( strlen(tb) + strlen( thisPtr->CoupleBuffer) >= (size_t)thisPtr->CoupleSize) 
    	{ 
    	 thisPtr->CoupleBuffer = (char *) realloc( thisPtr->CoupleBuffer,	
    		thisPtr->CoupleSize+1024); 
    	 if( !thisPtr->CoupleBuffer ) {return VI_ERROR_ALLOC;} 
    	 thisPtr->CoupleSize += 1024;
    	} 
      	
      strcat(thisPtr->CoupleBuffer,tb);
      
      return VI_SUCCESS;
    }
    hpe1445_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1445_coupleCGfreq
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This function is used to construct a coupled command consisting
 *            of the parameters set by the following SCPI commands:
 *           
 *            SOURce:FREQuency2:FIXed
 *            SOURce:LIST2:FREQuency
 *            SOURce:PM:SOURce
 *            SOURce:PM:STATe
 *            SOURce:ROSCillator:FREQuency:EXTernal
 *            SOURce:ROSCillator:SOURce
 *            ARM:SWEep:COUNt
 *            ARM:SWEep:SOURce
 *           
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument handle. In this case VI_NULL.
 * 
 * PARAM 2 : ViReal64 sourfreq2
 * IN        
 *            Sample rate, frequency, or hpe1445_CGFREQUNUSED  (= -1).
 *            If unused, then SOURce:FREQuency2 is not buffered.
 * 
 *      MAX = hpe1445_CGFREQF2MAX   42.9497e6
 *      MIN = hpe1445_CGFREQUNUSED   -1
 * 
 * PARAM 3 : ViString freqlist2
 * IN        
 *            List of sample rates or frequencies to be generated in mode
 *           'LIST' (set by hpe1445_coupleSourFreq1). VI_NULL if unused. List
 *           must be a comma separated ASCII string with no
 *            more than 256 entries. (e.g. "1.0,2.0").
 * 
 * PARAM 4 : ViString pmsource
 * IN        
 *            Source for phase modulation: "DPOR","INT","LBUS","VXI" or
 *           VI_NULL (unused).
 * 
 * PARAM 5 : ViString pmstate
 * IN        
 *            Enables ("ON") or disables ("OFF") phase modulation for sine
 *           waves. VI_NULL if unused.
 * 
 * PARAM 6 : ViReal64 roscfreq
 * IN        
 *            External reference frequency or hpe1445_CGFREQUNUSED (= -1).
 * 
 *      MAX = hpe1445_CGFREQROSCMAX   42.94967296e6
 *      MIN = hpe1445_CGFREQUNUSED   -1
 * 
 * PARAM 7 : ViString roscsour
 * IN        
 *            Reference oscillator source:
 *           "CLK10","ECLT0","ECLT1","EXT","INT1","INT2" or VI_NULL (unused).
 * 
 * PARAM 8 : ViInt32 armsweepcount
 * IN        
 *            Number of sweep arms accepted after INIT or
 *           hpe1445_CGFREQUNUSED (= -1).
 * 
 *      MAX = hpe1445_CGFREQSCOUNTMAX   2147483647
 *      MIN = hpe1445_CGFREQUNUSED   -1
 * 
 * PARAM 9 : ViString armsweepsour
 * IN        
 *            Source to start frequency sweep:
 *           "BUS","HOLD","IMM","LINK","TTLT0"-"TTLT7", or VI_NULL (unused).
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           the "hpe1413_error_message" routine.
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1445_coupleCGfreq(
  ViSession vi,
  ViReal64 sourfreq2,
  ViString freqlist2,
  ViString pmsource,
  ViString pmstate,
  ViReal64 roscfreq,
  ViString roscsour,
  ViInt32 armsweepcount,
  ViString armsweepsour)
{
    ViStatus errStatus = 0;
    struct hpe1445_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ hpe1445_LOG_STATUS( vi, 0, errStatus ); }

    hpe1445_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1445_CDE_INIT( "hpe1445_coupleCGfreq" );

    /* Perform Error Checking on Each Parameter */
    hpe1445_CHK_REAL_RANGE(sourfreq2
                          ,hpe1445_CGFREQUNUSED
                          ,hpe1445_CGFREQF2MAX
                          ,VI_ERROR_PARAMETER2);

    hpe1445_CHK_REAL_RANGE(roscfreq
                          ,hpe1445_CGFREQUNUSED
                          ,hpe1445_CGFREQROSCMAX
                          ,VI_ERROR_PARAMETER6);

    hpe1445_CHK_LONG_RANGE(armsweepcount
                          ,hpe1445_CGFREQUNUSED
                          ,hpe1445_CGFREQSCOUNTMAX
                          ,VI_ERROR_PARAMETER8);

    {
      char *lc;
      char *tb;
    
      lc = (char *) malloc(4096);
      tb = (char *) malloc(4096);
    
      if( !lc || !tb) return VI_ERROR_ALLOC;
    
      tb[0]=0;
      if( !thisPtr->CoupleBuffer ) 
      {
        free(lc); free(tb);
    	return hpe1445_ERR_NOBEGIN ;
      }
      
      if( sourfreq2 >= 0 )
      {
       sprintf(lc,":FREQ2:FIX %g;",sourfreq2);
       strcat(tb,lc);
      } else {if( sourfreq2 != -1) {free(lc); free(tb);return VI_ERROR_PARAMETER2;}}
      
      if( freqlist2 )
      {
       sprintf(lc,":LIST2:FREQ %s;",freqlist2);
       strcat(tb,lc);
    
      }
    
      
      if( pmsource)
      {
       sprintf(lc,":PM:SOUR %s;",pmsource);
       strcat(tb,lc);
    
      }
      
      if( pmstate )
      {
        sprintf(lc,":PM:STATE %s;",pmstate);
    	strcat(tb,lc);
    
      }
      
      if( roscfreq>=1 )
      {
        sprintf(lc,":ROSC:FREQ:EXT %g;",roscfreq);
    	strcat(tb,lc);
    
      }else {if( roscfreq != -1) {free(lc); free(tb);return VI_ERROR_PARAMETER6;}}
    
      if( roscsour )
      {
        sprintf(lc,":ROSC:SOUR %s;",roscsour);
    	strcat(tb,lc);
    
      }
    
      if( armsweepcount>=1 )
      {
        sprintf(lc,":ARM:SWE:COUN %d;",(int)armsweepcount);
    	strcat(tb,lc);
    
      }else {if( armsweepcount != -1) {free(lc); free(tb);return VI_ERROR_PARAMETER8;}}
    
      if( armsweepsour )
      {
        sprintf(lc,":ARM:SWE:SOUR %s;",armsweepsour);
    	strcat(tb,lc);
    
      }
      
      if( !strlen(tb) ) return VI_SUCCESS;
    
      if( strlen(tb) + strlen( thisPtr->CoupleBuffer) >= (size_t)thisPtr->CoupleSize) 
    	{ 
    	 thisPtr->CoupleBuffer = (char *) realloc( thisPtr->CoupleBuffer,	
    		thisPtr->CoupleSize+4096); 
    	 if( !thisPtr->CoupleBuffer ) {free(lc); free(tb);return VI_ERROR_ALLOC;} 
    	 thisPtr->CoupleSize += 4096;
    	} 
      	
      strcat(thisPtr->CoupleBuffer,tb);
      free(lc);
      free(tb);
      return VI_SUCCESS;
    }
    hpe1445_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1445_coupleCGvolt
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This function is used to construct a coupled command consisting
 *            of the parameters set by the following SCPI commands:
 *           
 *            OUTPut:IMPedance
 *            OUTPut:LOAD
 *            OUTPut:LOAD:AUTO
 *            SOURce:RAMP:POLarity
 *            SOURce:VOLTage:LEVel:IMMediate:AMPLitude:VOLTage
 *            SOURce:VOLTage:LEVel:IMMediate:OFFSet
 *           
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument handle. In this case VI_NULL.
 * 
 * PARAM 2 : ViInt32 impedance
 * IN        
 *            Set output impedance to 50 or 75 Ohms, allowed values are 50,
 *           75, or hpe1445_CGVOLT_UNUSED (=-20).
 * 
 *      Macro Name                 Value  Description
 *      -----------------------------------------------------------
 *      hpe1445_CGVOLT_UNUSED        -20  Unused
 *      hpe1445_Z_50                  50  50 Ohm
 *      hpe1445_Z_75                  75  75 Ohm
 * 
 * PARAM 3 : ViReal64 load
 * IN        
 *            Indicates if the actual load on the 50/75 ohm output is 50 or
 *           75 ohms, or is an open circuit. Values are 50, 75, 9.9e37 (open
 *           circuit), or hpe1445_CGVOLT_UNUSED (= -20).
 * 
 *      Macro Name                 Value  Description
 *      -----------------------------------------------------------
 *      hpe1445_CGVOLT_UNUSED        -20  Unused
 *      hpe1445_Z_50                  50  50 Ohm
 *      hpe1445_Z_75                  75  75 Ohm
 *      hpe1445_CGVOLT_INF        9.9e37  Infinity
 * 
 * PARAM 4 : ViString loadauto
 * IN        
 *            Indicates if output load value should be coupled to (track) the
 *           output impedance value. Thus, changing the impedance changes the
 *           load. Values are "ON" (enabled), "OFF" (disabled), or VI_NULL
 *           (unused).
 * 
 * PARAM 5 : ViString ramppol
 * IN        
 *            Selects the polarity of the ramp, triangle, or square wave.
 *            Values are "INV","NORM", or VI_NULL (unused).
 * 
 * PARAM 6 : ViReal64 ampl
 * IN        
 *            Sets the output amplitude, or Hpe1445_CGVOLT_UNUSED (= -20)
 *            if unused.
 *           
 * 
 *      MAX = hpe1445_CGVOLT_AMPLMAX   12.05
 *      MIN = hpe1445_CGVOLT_UNUSED   -20
 * 
 * PARAM 7 : ViReal64 offset
 * IN        
 *            Sets the output offset voltage, or Hpe1445_CGVOLT_UNUSED (=
 *           -20) if unused.
 * 
 *      MAX = hpe1445_CGVOLT_AMPLMAX   12.05
 *      MIN = hpe1445_CGVOLT_UNUSED   -20
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           the "hpe1413_error_message" routine.
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1445_coupleCGvolt(
  ViSession vi,
  ViInt32 impedance,
  ViReal64 load,
  ViString loadauto,
  ViString ramppol,
  ViReal64 ampl,
  ViReal64 offset)
{
    ViStatus errStatus = 0;
    struct hpe1445_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ hpe1445_LOG_STATUS( vi, 0, errStatus ); }

    hpe1445_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1445_CDE_INIT( "hpe1445_coupleCGvolt" );

    /* Perform Error Checking on Each Parameter */
    hpe1445_CHK_REAL_RANGE(ampl
                          ,hpe1445_CGVOLT_UNUSED
                          ,hpe1445_CGVOLT_AMPLMAX
                          ,VI_ERROR_PARAMETER6);

    hpe1445_CHK_REAL_RANGE(offset
                          ,hpe1445_CGVOLT_UNUSED
                          ,hpe1445_CGVOLT_AMPLMAX
                          ,VI_ERROR_PARAMETER7);

    {
      char lc[512];
      char tb[512];
    
      
      tb[0]=0;
      if( !thisPtr->CoupleBuffer ) 
      {
        
    	return hpe1445_ERR_NOBEGIN ;
      }
      
      if( impedance == 50 || impedance == 75 )
      {
       sprintf(lc,":OUTP:IMP %d;",(int)impedance);
       strcat(tb,lc);
      }else if( impedance != hpe1445_CGVOLT_UNUSED) return VI_ERROR_PARAMETER2;
      
      if( load == 50 || load == 75 || load == hpe1445_CGVOLT_INF )
      {
       sprintf(lc,":OUTP:LOAD %g;",load);
       strcat(tb,lc);
    
      }else if( impedance != hpe1445_CGVOLT_UNUSED) return VI_ERROR_PARAMETER3;
    
    
      
      if( loadauto)
      {
       sprintf(lc,":OUTP:LOAD:AUTO %s;",loadauto);
       strcat(tb,lc);
    
      }
      
      if( ramppol )
      {
        sprintf(lc,":RAMP:POL %s;",ramppol);
    	strcat(tb,lc);
    
      }
      
      if( ampl >= -12.05 )
      {
        sprintf(lc,":VOLT %g;",ampl);
    	strcat(tb,lc);
    
      }else if( ampl != hpe1445_CGVOLT_UNUSED) return VI_ERROR_PARAMETER6;
    
      if( offset >= -12.05)
      {
        sprintf(lc,":VOLT:OFFS %g;",offset);
    	strcat(tb,lc);
    
      }	else if( offset != hpe1445_CGVOLT_UNUSED) return VI_ERROR_PARAMETER7;
    
    
      if( !strlen(tb) ) return VI_SUCCESS;
    
      if( strlen(tb) + strlen( thisPtr->CoupleBuffer) >= (size_t)thisPtr->CoupleSize) 
    	{ 
    	 thisPtr->CoupleBuffer = (char *) realloc( thisPtr->CoupleBuffer,	
    		thisPtr->CoupleSize+1024); 
    	 if( !thisPtr->CoupleBuffer ) {return VI_ERROR_ALLOC;} 
    	 thisPtr->CoupleSize += 1024;
    	} 
      	
      strcat(thisPtr->CoupleBuffer,tb);
      
      return VI_SUCCESS;
    }
    hpe1445_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1445_coupleEnd
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This function closes and sends the coupled command started with
 *            the hpe1445_coupleBegin function.
 *           
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument handle. In this case VI_NULL.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           the "hpe1413_error_message" routine.
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1445_coupleEnd(
  ViSession vi)
{
    ViStatus errStatus = 0;
    struct hpe1445_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ hpe1445_LOG_STATUS( vi, 0, errStatus ); }

    hpe1445_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1445_CDE_INIT( "hpe1445_coupleEnd" );

    /* Perform Error Checking on Each Parameter */
    {
      if( !thisPtr->CoupleBuffer || !strlen(thisPtr->CoupleBuffer) ) 
      {
    	return hpe1445_ERR_NOBEGIN;
      }
      
      thisPtr->CoupleBuffer[strlen(thisPtr->CoupleBuffer)-1]='\n';
    
      errStatus = viPrintf(vi,thisPtr->CoupleBuffer);
      thisPtr->CoupleBuffer[strlen(thisPtr->CoupleBuffer)-1]=';';
      if( errStatus < VI_SUCCESS)hpe1445_LOG_STATUS( vi, thisPtr, errStatus ); 
    }
    hpe1445_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1445_coupleSourFreq1
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This function is used to construct a coupled command consisting
 *            of the parameters set by the following SCPI commands:
 *           
 *            SOURce:FREQuency1:CENTer
 *            SOURce:FREQuency1:FIXed
 *            SOURce:FREQuency1:FSKey
 *            SOURce:FREQuency1:SOURce
 *            SOURce:FREQuency1:MODE
 *            SOURce:FREQuency1:RANGe
 *            SOURce:FREQuency1:SPAN
 *            SOURce:FREQuency1:STARt
 *            SOURce:FREQuency1:STOP
 *           
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument handle. In this case VI_NULL.
 * 
 * PARAM 2 : ViReal64 center
 * IN        
 *            Center frequency, or hpe1445_FREQ1UNUSED  (= -1)  If unused
 *           then SOUR:FREQ1:CENTER is not buffered.
 * 
 *      MAX = hpe1445_FREQ1MAX   42.94967296e6
 *      MIN = hpe1445_FREQ1_UNUSED   -1
 * 
 * PARAM 3 : ViReal64 fixed
 * IN        
 *            Sample rate, or hpe1445_FREQ1UNUSED  (= -1)  If unused then
 *            SOUR:FREQ1 is not buffered.
 * 
 *      MAX = hpe1445_FREQ1MAX   42.94967296e6
 *      MIN = hpe1445_FREQ1_UNUSED   -1
 * 
 * PARAM 4 : ViReal64 fsk1
 * IN        
 *            Frequency-shift keying frequency 1, or hpe1445_FREQ1UNUSED  (=
 *           -1). If unused, then SOUR:FREQ1:FSK is not buffered. Also,
 *           OnTrigSelector is ignored no matter its value.
 * 
 *      MAX = hpe1445_FREQ1MAX   42.94967296e6
 *      MIN = hpe1445_FREQ1_UNUSED   -1
 * 
 * PARAM 5 : ViReal64 fsk2
 * IN        
 *            Frequency-shift keying frequency 2 If fsk1 is unused,
 *            then this parameter is ignored and must be zero.  If fsk1 is
 *           used, this parameter MUST be specified.
 * 
 *      MAX = hpe1445_FREQ1_FSK2_MAX   42.94967296e6
 *      MIN = hpe1445_FREQ1_FSK2_MIN   0
 * 
 * PARAM 6 : ViString fskSour
 * IN        
 *            FSK source, or VI_NULL  if unused.  If VI_NULL, then
 *            SOUR:FREQ1:FSK:SOUR is not buffered.  The values for this
 *            parameter are "EXT","TTLT0" - "TTLT7"
 * 
 * PARAM 7 : ViString mode
 * IN        
 *            Frequency mode or VI_NULL if unused. If VI_NULL, then
 *            SOUR:FREQ1:MODE is not buffered.  The values for this
 *            parameter are "CW","FIX","FSK","LIST","SWE" or VI_NULL.
 * 
 * PARAM 8 : ViReal64 range
 * IN        
 *            Enable/disable frequency doubling for non sine waves, or
 *           hpe1445_FREQ1UNUSED (= -1).  If unused, then SOUR:FREQ1:RANGE is
 *           not buffered.
 * 
 *      MAX = hpe1445_FREQ1MAX   42.94967296e6
 *      MIN = hpe1445_FREQ1_UNUSED   -1
 * 
 * PARAM 9 : ViReal64 span
 * IN        
 *            Sample rate or waveform frequency span for a swept waveform, or
 *            hpe1445_FREQ1UNUSED (=-1)  If unused, then SOUR:FREQ1:SPAN is
 *           not buffered.
 * 
 *      MAX = hpe1445_FREQ1MAX   42.94967296e6
 *      MIN = hpe1445_FREQ1_UNUSED   -1
 * 
 * PARAM 10IN : ViReal64 start
 * IN        
 *            Starting sample rate or frequency for a swept waveform, or
 *            hpe1445_FREQ1UNUSED (= -1)  If unused, then SOUR:FREQ1:START is
 *           not buffered.
 * 
 *      MAX = hpe1445_FREQ1MAX   42.94967296e6
 *      MIN = hpe1445_FREQ1_UNUSED   -1
 * 
 * PARAM 11IN : ViReal64 stop
 * IN        
 *            Stopping sample rate or frequency for a swept waveform, or
 *            hpe1445_FREQ1UNUSED (= -1)  If unused, then SOUR:FREQ1:STOP is
 *           not buffered.
 * 
 *      MAX = hpe1445_FREQ1MAX   42.94967296e6
 *      MIN = hpe1445_FREQ1_UNUSED   -1
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           the "hpe1413_error_message" routine.
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1445_coupleSourFreq1(
  ViSession vi,
  ViReal64 center,
  ViReal64 fixed,
  ViReal64 fsk1,
  ViReal64 fsk2,
  ViString fskSour,
  ViString mode,
  ViReal64 range,
  ViReal64 span,
  ViReal64 start,
  ViReal64 stop)
{
    ViStatus errStatus = 0;
    struct hpe1445_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ hpe1445_LOG_STATUS( vi, 0, errStatus ); }

    hpe1445_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1445_CDE_INIT( "hpe1445_coupleSourFreq1" );

    /* Perform Error Checking on Each Parameter */
    hpe1445_CHK_REAL_RANGE(center
                          ,hpe1445_FREQ1_UNUSED
                          ,hpe1445_FREQ1MAX
                          ,VI_ERROR_PARAMETER2);

    hpe1445_CHK_REAL_RANGE(fixed
                          ,hpe1445_FREQ1_UNUSED
                          ,hpe1445_FREQ1MAX
                          ,VI_ERROR_PARAMETER3);

    hpe1445_CHK_REAL_RANGE(fsk1
                          ,hpe1445_FREQ1_UNUSED
                          ,hpe1445_FREQ1MAX
                          ,VI_ERROR_PARAMETER4);

    hpe1445_CHK_REAL_RANGE(fsk2
                          ,hpe1445_FREQ1_FSK2_MIN
                          ,hpe1445_FREQ1_FSK2_MAX
                          ,VI_ERROR_PARAMETER5);

    hpe1445_CHK_REAL_RANGE(range
                          ,hpe1445_FREQ1_UNUSED
                          ,hpe1445_FREQ1MAX
                          ,VI_ERROR_PARAMETER8);

    hpe1445_CHK_REAL_RANGE(span
                          ,hpe1445_FREQ1_UNUSED
                          ,hpe1445_FREQ1MAX
                          ,hpe1445_INSTR_ERROR_PARAMETER9);

    hpe1445_CHK_REAL_RANGE(start
                          ,hpe1445_FREQ1_UNUSED
                          ,hpe1445_FREQ1MAX
                          ,hpe1445_INSTR_ERROR_PARAMETER10);

    hpe1445_CHK_REAL_RANGE(stop
                          ,hpe1445_FREQ1_UNUSED
                          ,hpe1445_FREQ1MAX
                          ,hpe1445_INSTR_ERROR_PARAMETER11);

    {
      char lc[512];
      char tb[512];
      tb[0]=0;
      if( !thisPtr->CoupleBuffer ) 
      {
    	return hpe1445_ERR_NOBEGIN ;
      }
      
      if( center >= 0 )
      {
       sprintf(lc,":FREQ1:CENT %g;",center);
       strcat(tb,lc);
      }else {if( center != -1) return VI_ERROR_PARAMETER2;}
    
      
      if( fixed >= 0)
      {
       sprintf(lc,":FREQ1:FIX %g;",fixed);
       strcat(tb,lc);
    
      }else {if( fixed != -1) return VI_ERROR_PARAMETER3;}
    
      
      if( fsk1 >= 0)
      {
       sprintf(lc,":FREQ1:FSK %g,%g;",fsk1,fsk2);
       strcat(tb,lc);
    
      }else {if( fsk1 != -1) return VI_ERROR_PARAMETER4;}
    
      
      if( fskSour )
      {
        sprintf(lc,":FREQ1:FSK:SOUR %s;",fskSour);
    	strcat(tb,lc);
    
      }
      
      if( mode )
      {
        sprintf(lc,":FREQ1:MODE %s;",mode);
    	strcat(tb,lc);
    
      }
      
      if( range >= 0)
      {
        sprintf(lc,":FREQ1:RANG %g;",range);
        strcat(tb,lc);
    
      }	else {if( range != -1) return VI_ERROR_PARAMETER8;}
    
      
      if( span >= 0)
      {
       sprintf(lc,":FREQ1:SPAN %g;",span);
       strcat(tb,lc);
    
      }else {if( span != -1) return hpe1445_INSTR_ERROR_PARAMETER9;}
    
      
      if( start >= 0)
      {
      	sprintf(lc,":FREQ1:STAR %g;",start);
    	strcat(tb,lc);
    
      }else {if( start != -1) return hpe1445_INSTR_ERROR_PARAMETER10;}
    
      
      if( stop >= 0 )
      {
      	sprintf(lc,":FREQ1:STOP %g;",stop);
    	strcat(tb,lc);
    
      }	else {if( stop != -1) return hpe1445_INSTR_ERROR_PARAMETER11;}
    
        
      if( strlen(tb) + strlen( thisPtr->CoupleBuffer) >= (size_t)thisPtr->CoupleSize) 
    	{ 
    	 thisPtr->CoupleBuffer = (char *) realloc( thisPtr->CoupleBuffer,	
    		thisPtr->CoupleSize+1024); 
    	 if( !thisPtr->CoupleBuffer ) return VI_ERROR_ALLOC; 
    	 thisPtr->CoupleSize += 1024;
    	} 
    	
      strcat(thisPtr->CoupleBuffer,tb);
      return VI_SUCCESS;
    }
    hpe1445_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1445_coupleSourSweep
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This function is used to construct a coupled command consisting
 *            of the parameters set by the following SCPI commands:
 *           
 *            SOURce:SWEep:COUNt
 *            SOURce:SWEep:DIRection
 *            SOURce:SWEep:POINts
 *            SOURce:SWEep:SPACing
 *            SOURce:SWEep:TIMe
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument handle. In this case VI_NULL.
 * 
 * PARAM 2 : ViInt32 count
 * IN        
 *            Number of sweeps or repetitions of a frequency, or
 *           hpe1445_SWEEPUNUSED  (= -1). If unused, then SOURce:SWEep:COUNt
 *           is not buffered.
 * 
 *      MAX = hpe1445_SWEEPCOUNTCMAX   2147483647
 *      MIN = hpe1445_SWEEPUNUSED   -1
 * 
 * PARAM 3 : ViString direction
 * IN        
 *            Sweep direction, "DOWN", "UP", or VI_NULL (unused).  If unused,
 *           then SOURce:SWEep:DIRection is not buffered.
 * 
 * PARAM 4 : ViInt32 points
 * IN        
 *            Number of points in a frequency sweep, or hpe1445_SWEEPUNUSED 
 *           (= -1). If unused, then SOUR:SWEEP:POINTS is not buffered.
 * 
 *      MAX = hpe1445_SWEEPPOINTSMAX   1073741824
 *      MIN = hpe1445_SWEEPUNUSED   -1
 * 
 * PARAM 5 : ViString spacing
 * IN        
 *            Selects linear ("LIN") or log ("LOG") sweep mode. VI_NULL if
 *            unused.
 * 
 * PARAM 6 : ViReal64 time
 * IN        
 *            Duration of sweep or frequency or hpe1445_SWEEPUNUSED (= -1).
 * 
 *      MAX = hpe1445_REALMAX   1e39
 *      MIN = hpe1445_SWEEPUNUSED   -1
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           the "hpe1413_error_message" routine.
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1445_coupleSourSweep(
  ViSession vi,
  ViInt32 count,
  ViString direction,
  ViInt32 points,
  ViString spacing,
  ViReal64 time)
{
    ViStatus errStatus = 0;
    struct hpe1445_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ hpe1445_LOG_STATUS( vi, 0, errStatus ); }

    hpe1445_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1445_CDE_INIT( "hpe1445_coupleSourSweep" );

    /* Perform Error Checking on Each Parameter */
    hpe1445_CHK_LONG_RANGE(count
                          ,hpe1445_SWEEPUNUSED
                          ,hpe1445_SWEEPCOUNTCMAX
                          ,VI_ERROR_PARAMETER2);

    hpe1445_CHK_LONG_RANGE(points
                          ,hpe1445_SWEEPUNUSED
                          ,hpe1445_SWEEPPOINTSMAX
                          ,VI_ERROR_PARAMETER4);

    hpe1445_CHK_REAL_RANGE(time
                          ,hpe1445_SWEEPUNUSED
                          ,hpe1445_REALMAX
                          ,VI_ERROR_PARAMETER6);

    {
      char lc[512];
      char tb[512];
      tb[0]=0;
      if( !thisPtr->CoupleBuffer ) 
      {
    	return hpe1445_ERR_NOBEGIN ;
      }
      
      if( count >= 0 )
      {
       sprintf(lc,":SWE:COUN %d;",(int)count);
       strcat(tb,lc);
      } else {if( count != -1) return VI_ERROR_PARAMETER2;}
      
      if( direction)
      {
       sprintf(lc,":SWE:DIR %s;",direction);
       strcat(tb,lc);
    
      }
    
      
      if( points >= 0)
      {
       sprintf(lc,":SWE:POIN %d;",(int)points);
       strcat(tb,lc);
    
      }else {if( points != -1) return VI_ERROR_PARAMETER4;}
    
      
      if( spacing )
      {
        sprintf(lc,":SWE:SPAC %s;",spacing);
    	strcat(tb,lc);
    
      }
      
      if( time>=0 )
      {
        sprintf(lc,":SWE:TIME %g;",time);
    	strcat(tb,lc);
    
      }else {if( time != -1) return VI_ERROR_PARAMETER6;}
    
      
      if( !strlen(tb) ) return VI_SUCCESS;
    
      if( strlen(tb) + strlen( thisPtr->CoupleBuffer) >= (size_t)thisPtr->CoupleSize) 
    	{ 
    	 thisPtr->CoupleBuffer = (char *) realloc( thisPtr->CoupleBuffer,	
    		thisPtr->CoupleSize+1024); 
    	 if( !thisPtr->CoupleBuffer ) return VI_ERROR_ALLOC; 
    	 thisPtr->CoupleSize += 1024;
    	} 
    	
      strcat(thisPtr->CoupleBuffer,tb);
      return VI_SUCCESS;
    }
    hpe1445_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1445_coupleTrig
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This function is used to construct a coupled command consisting
 *            of the parameters set by the following SCPI commands:
 *           
 *            TRIGger:STARt:GATE:POLarity
 *            TRIGger:STARt:GATE:SOURce
 *            TRIGger:STARt:GATE:STATe
 *            TRIG:STARt:SOURce
 *            TRIGger:STOP:SLOPe
 *            TRIGger:STOP:SOURce
 *            TRIGger:SWEep:SOURce
 *            TRIGger:SWEep:TIMer
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument handle. In this case VI_NULL.
 * 
 * PARAM 2 : ViString polarity
 * IN        
 *            Front Panel 'Stop Trig/FSK/Gate In' polarity, "INV", "NORM", or
 *           VI_NULL (unused).
 * 
 * PARAM 3 : ViString gatesour
 * IN        
 *            Source for TRIGger:STARt subsystem: "EXT","TTLT0"-"TTLT7", or
 *           VI_NULL (unused).
 * 
 * PARAM 4 : ViString gatestate
 * IN        
 *            Enable ("ON") / disable ("OFF") sample gating, or VI_NULL
 *           (unused).
 * 
 * PARAM 5 : ViString startsour
 * IN        
 *            Source that advances the waveform to  the next sample point:
 *            "BUS","ECTL0",ECTL1",EXT","HOLD","INT1","INT2",
 *           "TTLT0"-"TTLT7", or VI_NULL (unused).
 * 
 * PARAM 6 : ViString slope
 * IN        
 *            Selects the rising ("POS") or falling ("NEG") edge on
 *            the front panel 'Stop/FSK/Gate In' BNC which terminates
 *            the current start arm cycle at the end of the current waveform.
 *            VI_NULL  is specified if unused.
 * 
 * PARAM 7 : ViString stopsour
 * IN        
 *            Selects source that can terminate the current start arm cycle:
 *            "BUS","EXT","HOLD","TTLT0"-"TTLT7", or VI_NULL (unused).
 * 
 * PARAM 8 : ViString sweepsour
 * IN        
 *            Select source that advances frequency sweep to the next
 *           frequency: "BUS","HOLD","LINK","TIM","TTLT0"-"TTLT7", or VI_NULL
 *           (unused).
 * 
 * PARAM 9 : ViReal64 period
 * IN        
 *            Selects the time between frequency values for sweep or
 *           frequency when TRIGger:SWEep:SOURce is 'TIMer', or
 *           Hpe1445_TRIGUNUSED (= -1) if unused.
 * 
 *      MAX = hpe1445_TRIGPERIODMAX   4.19430375
 *      MIN = hpe1445_TRIGUNUSED   -1
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           the "hpe1413_error_message" routine.
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1445_coupleTrig(
  ViSession vi,
  ViString polarity,
  ViString gatesour,
  ViString gatestate,
  ViString startsour,
  ViString slope,
  ViString stopsour,
  ViString sweepsour,
  ViReal64 period)
{
    ViStatus errStatus = 0;
    struct hpe1445_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ hpe1445_LOG_STATUS( vi, 0, errStatus ); }

    hpe1445_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1445_CDE_INIT( "hpe1445_coupleTrig" );

    /* Perform Error Checking on Each Parameter */
    hpe1445_CHK_REAL_RANGE(period
                          ,hpe1445_TRIGUNUSED
                          ,hpe1445_TRIGPERIODMAX
                          ,hpe1445_INSTR_ERROR_PARAMETER9);

    {
      char lc[512];
      char tb[512];
    
      
      tb[0]=0;
      if( !thisPtr->CoupleBuffer ) 
      {
        
    	return hpe1445_ERR_NOBEGIN ;
      }
      
      if( polarity )
      {
       sprintf(lc,":TRIG:GATE:POL %s;",polarity);
       strcat(tb,lc);
      }
      
      if( gatesour )
      {
       sprintf(lc,":TRIG:GATE:SOUR %s;",gatesour);
       strcat(tb,lc);
    
      }
    
      
      if( gatestate)
      {
       sprintf(lc,":TRIG:GATE:STAT %s;",gatestate);
       strcat(tb,lc);
    
      }
      
      if( startsour )
      {
        sprintf(lc,":TRIG:SOUR %s;",startsour);
    	strcat(tb,lc);
    
      }
      
      if( slope )
      {
        sprintf(lc,":TRIG:STOP:SLOP %s;",slope);
    	strcat(tb,lc);
    
      }
    
      if( stopsour )
      {
        sprintf(lc,":TRIG:STOP:SOUR %s;",stopsour);
    	strcat(tb,lc);
    
      }
    
      if( sweepsour )
      {
        sprintf(lc,":TRIG:SWE:SOUR %s;",sweepsour);
    	strcat(tb,lc);
    
      }
    
      if( period >= 1.25e-3 )
      {
        sprintf(lc,":TRIG:SWE:TIM %g;",period);
    	strcat(tb,lc);
    
      }else if( period != -1) return hpe1445_INSTR_ERROR_PARAMETER9;
      
      if( !strlen(tb) ) return VI_SUCCESS;
    
      if( strlen(tb) + strlen( thisPtr->CoupleBuffer) >= (size_t)thisPtr->CoupleSize) 
    	{ 
    	 thisPtr->CoupleBuffer = (char *) realloc( thisPtr->CoupleBuffer,	
    		thisPtr->CoupleSize+1024); 
    	 if( !thisPtr->CoupleBuffer ) {return VI_ERROR_ALLOC;} 
    	 thisPtr->CoupleSize += 1024;
    	} 
      	
      strcat(thisPtr->CoupleBuffer,tb);
      
      return VI_SUCCESS;
    }
    hpe1445_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1445_createSeg
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This function creates a waveform segment in waveform segment
 *           memory.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument handle. In this case VI_NULL.
 * 
 * PARAM 2 : ViString name
 * IN        
 *            Segment name.
 * 
 * PARAM 3 : ViInt32 size
 * IN        
 *            Number of points in the segment.
 * 
 *      MAX = hpe1445_SEGNO_MAX   262144
 *      MIN = hpe1445_SEGNO_MIN   8
 * 
 * PARAM 4 : ViReal64 data[]
 * IN        
 *            Segment points.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1413_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1445_createSeg(
  ViSession vi,
  ViString name,
  ViInt32 size,
  ViReal64 data[])
{
    ViStatus errStatus = 0;
    struct hpe1445_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ hpe1445_LOG_STATUS( vi, 0, errStatus ); }

    hpe1445_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1445_CDE_INIT( "hpe1445_createSeg" );

    /* Perform Error Checking on Each Parameter */
    hpe1445_CHK_LONG_RANGE(size
                          ,hpe1445_SEGNO_MIN
                          ,hpe1445_SEGNO_MAX
                          ,VI_ERROR_PARAMETER3);

    { 
      struct ListStruct *p;
    
      p=thisPtr->ListHead;
      while(p)
      {
    	if( strcmp(name,p->name) == 0 )
    	{
    		if( !(p->data=realloc( p->data,size*sizeof(ViReal64))) )
    				return VI_ERROR_ALLOC;
    		p->size = size;
    		memcpy(p->data,data,size*sizeof(ViReal64));
    		return VI_SUCCESS;
    	}
    	
    	p=p->next;
      }
    
      if( thisPtr->ListTail)
      {
    	thisPtr->ListTail->next =	
    		(struct ListStruct *)malloc(sizeof(struct ListStruct));
    	if( !thisPtr->ListTail->next) return VI_ERROR_ALLOC;
    	p=thisPtr->ListTail->next;
    	p->next = NULL;
    	p->MarkerList=NULL;
    	thisPtr->ListTail = p;
      }
      else
        {
    	 thisPtr->ListTail =(struct ListStruct *)malloc(sizeof(struct ListStruct));
     	 if( !thisPtr->ListTail) return VI_ERROR_ALLOC;
    	 thisPtr->ListTail->next = NULL;
    	 thisPtr->ListTail->MarkerList=NULL;
    	 p=thisPtr->ListTail;
    	 thisPtr->ListHead = p;
    	}
        
      strcpy(p->name,name);
      p->size = size;
      p->data = malloc(size*sizeof(ViReal64));
      if( !p->data ) return VI_ERROR_ALLOC;
      
      memcpy(p->data,data,size*sizeof(ViReal64));
      
      return VI_SUCCESS;
    }
    hpe1445_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1445_fsk1
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This function sets up frequency-shift keying for a sine wave
 *           signal. The frequency is shifted based on the signal applied to
 *           either the front panel Fsk BNC, or the specified VXI backplane
 *           TTL trigger line.
 *           
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument handle. In this case VI_NULL.
 * 
 * PARAM 2 : ViReal64 freq1
 * IN        
 *            First FSK frequency.
 * 
 *      MAX = hpe1445_FSKF1_MAX   42.94e6
 *      MIN = hpe1445_FSKF1_MIN   0
 * 
 * PARAM 3 : ViReal64 freq2
 * IN        
 *            Second FSK frequency.
 * 
 *      MAX = hpe1445_FSKF2_MAX   42.94e6
 *      MIN = hpe1445_FSKF2_MIN   0
 * 
 * PARAM 4 : ViReal64 ampl
 * IN        
 *            Signal amplitude in volts peak.
 * 
 *      MAX = hpe1445_FSKAMPL_MAX   12.05
 *      MIN = hpe1445_FSKAMPL_MIN   0
 * 
 * PARAM 5 : ViString fsksource
 * IN        
 *            Frequency-shift keying source. 'EXT' (the stop/trig/FSK/Gate In
 *           BNC) or 'TTLT0' - 'TTLT7' (one of the VXI TTL trigger lines).
 * 
 * PARAM 6 : ViInt32 impedance
 * IN        
 *            Sets the output impedance to 50 or 75 ohms.
 * 
 *      Macro Name        Value  Description
 *      -----------------------------------------------------------
 *      hpe1445_Z_50         50  50 Ohm
 *      hpe1445_Z_75         75  75 Ohm
 * 
 * PARAM 7 : ViString load
 * IN        
 *            Sets the output load. 'MIN' (50), 'MAX' (75),  'INF'
 *           (infinity).
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           the "hpe1413_error_message" routine.
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1445_fsk1(
  ViSession vi,
  ViReal64 freq1,
  ViReal64 freq2,
  ViReal64 ampl,
  ViString fsksource,
  ViInt32 impedance,
  ViString load)
{
    ViStatus errStatus = 0;
    struct hpe1445_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ hpe1445_LOG_STATUS( vi, 0, errStatus ); }

    hpe1445_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1445_CDE_INIT( "hpe1445_fsk1" );

    /* Perform Error Checking on Each Parameter */
    hpe1445_CHK_REAL_RANGE(freq1
                          ,hpe1445_FSKF1_MIN
                          ,hpe1445_FSKF1_MAX
                          ,VI_ERROR_PARAMETER2);

    hpe1445_CHK_REAL_RANGE(freq2
                          ,hpe1445_FSKF2_MIN
                          ,hpe1445_FSKF2_MAX
                          ,VI_ERROR_PARAMETER3);

    hpe1445_CHK_REAL_RANGE(ampl
                          ,hpe1445_FSKAMPL_MIN
                          ,hpe1445_FSKAMPL_MAX
                          ,VI_ERROR_PARAMETER4);

    { 
      errStatus = viPrintf(vi,":ABORT\n");
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
      errStatus = viPrintf(vi,":VOLT:UNIT V\n;");
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
      errStatus = viPrintf(vi,":SOUR:ROSC:SOUR INT1;");
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
      errStatus = viPrintf(vi,":TRIG:STAR:SOUR INT1;");
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
      errStatus = viPrintf(vi,":SOUR:FREQ1:FSK %g,%g;",freq1,freq2);
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
      errStatus = viPrintf(vi,":SOUR:FREQ1:FSK:SOUR %s;",fsksource);
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
      if( impedance == 50 || impedance == 75)
      {
      errStatus = viPrintf(vi,":OUTP:IMP %d;",(int)impedance);
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
      }else {hpe1445_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER6 );}
    
      errStatus = viPrintf(vi,":OUTP:LOAD %s;",load);
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
      errStatus = viPrintf(vi,":SOUR:FUNC:SHAP SIN;");
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
     
      errStatus = viPrintf(vi,":SOUR:FREQ1:MODE FSK;");
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
      errStatus = viPrintf(vi,":SOUR:VOLT:LEV:IMM:AMPL %g\n",ampl);
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
      
    
      errStatus = viPrintf(vi,":INIT:IMM\n");
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
    }
    hpe1445_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1445_marker
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  hpe1445_marker sets a marker list for a waveform segment in
 *            an arbitrary waveform.  CreateSeg must have been called with
 *           this segment name before calling hpe1445_marker.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument handle. In this case VI_NULL.
 * 
 * PARAM 2 : ViString segment
 * IN        
 *            Name of waveform segment.
 * 
 * PARAM 3 : ViInt16 markerlist[]
 * IN        
 *            List of values either '1' or '0'.  There MUST be as many
 *           entries in this list as there are points in the wave segment
 *           'segment'. A '1' sets the point to output the marker.  Can be
 *           VI_NULL in which case 'segment' will no longer have a maker list
 *           associated with it.  Doing this for all segments in a waveform
 *           will essentially 'turn off' markers (which is the default).
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           the "hpe1413_error_message" routine.
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1445_marker(
  ViSession vi,
  ViString segment,
  ViInt16 markerlist[])
{
    ViStatus errStatus = 0;
    struct hpe1445_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ hpe1445_LOG_STATUS( vi, 0, errStatus ); }

    hpe1445_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1445_CDE_INIT( "hpe1445_marker" );

    /* Perform Error Checking on Each Parameter */
    { 
        struct ListStruct *p;
    	int i;
    
        p=thisPtr->ListHead;
    	while(p)
    		{
    			if( strcmp(segment,p->name) == 0)
    			{
    			 
    				if(!p->MarkerList ) p->MarkerList = (char *) malloc(p->size);
    				if( !p->MarkerList ) return VI_ERROR_ALLOC;
    
    				if( !markerlist ) 
    				{
    				  if( p->MarkerList) free(p->MarkerList);
    				  p->MarkerList=NULL;
    				  break;
    				}
    				for(i=0; i<p->size; i++)
    					{
    					  p->MarkerList[i]=(char)markerlist[i];			    
    					}
    				break;
    			}
    			p=p->next;
    		}
    
    	if( !p ) return hpe1445_ERR_SEGNAME;
     	return VI_SUCCESS;
    }
    hpe1445_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1445_outpState
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Closes or opens the Agilent E1445's output relay to enable or
 *           disable the analog output.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument handle. In this case VI_NULL.
 * 
 * PARAM 2 : ViString state
 * IN        
 *            'ON' = enable analog output, 'OFF' = disable.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           the "hpe1413_error_message" routine.
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1445_outpState(
  ViSession vi,
  ViString state)
{
    ViStatus errStatus = 0;
    struct hpe1445_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ hpe1445_LOG_STATUS( vi, 0, errStatus ); }

    hpe1445_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1445_CDE_INIT( "hpe1445_outpState" );

    /* Perform Error Checking on Each Parameter */
    { 
        errStatus = viPrintf(vi,"OUTP %s\n",state);
    	if( errStatus < VI_SUCCESS) hpe1445_LOG_STATUS( vi, thisPtr, errStatus );
    
    }
    hpe1445_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1445_sinWave
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This function generates a sine wave.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument handle. In this case VI_NULL.
 * 
 * PARAM 2 : ViReal64 freq
 * IN        
 *            Signal frequency.
 * 
 *      MAX = hpe1445_SIN_FREQ_MAX   42.94e6
 *      MIN = hpe1445_SIN_FREQ_MIN   0
 * 
 * PARAM 3 : ViReal64 ampl
 * IN        
 *            Signal amplitude (volts peak).
 * 
 *      MAX = hpe1445_SIN_AMPL_MAX   12.05
 *      MIN = hpe1445_SIN_AMPL_MIN   0
 * 
 * PARAM 4 : ViInt32 impedance
 * IN        
 *            Output impedance: 50 or 75 ohms.
 * 
 *      Macro Name        Value  Description
 *      -----------------------------------------------------------
 *      hpe1445_Z_50         50  50 Ohm
 *      hpe1445_Z_75         75  75 Ohm
 * 
 * PARAM 5 : ViString load
 * IN        
 *            Output load. "MIN" (50), "MAX" (75) or "INF" (infinity).
 *            Use infinity to get the correct amplitude for an open circuit.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           the "hpe1413_error_message" routine.
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1445_sinWave(
  ViSession vi,
  ViReal64 freq,
  ViReal64 ampl,
  ViInt32 impedance,
  ViString load)
{
    ViStatus errStatus = 0;
    struct hpe1445_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ hpe1445_LOG_STATUS( vi, 0, errStatus ); }

    hpe1445_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1445_CDE_INIT( "hpe1445_sinWave" );

    /* Perform Error Checking on Each Parameter */
    hpe1445_CHK_REAL_RANGE(freq
                          ,hpe1445_SIN_FREQ_MIN
                          ,hpe1445_SIN_FREQ_MAX
                          ,VI_ERROR_PARAMETER2);

    hpe1445_CHK_REAL_RANGE(ampl
                          ,hpe1445_SIN_AMPL_MIN
                          ,hpe1445_SIN_AMPL_MAX
                          ,VI_ERROR_PARAMETER3);

    {  
       errStatus = viPrintf(vi,":ABORT\n");
       if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
       errStatus = viPrintf(vi,":VOLT:UNIT V\n;");
       if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
       
       errStatus = viPrintf(vi,":FREQ1:MODE FIX;");
       if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
       errStatus = viPrintf(vi,":SOUR:FREQ:FIX %f;",freq);
       if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
       errStatus = viPrintf(vi,":SOUR:FUNC:SHAPE SIN;");
       if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
       if( impedance == 50 || impedance == 75)
       {
       errStatus = viPrintf(vi,":OUTP:IMP %d;",(int)impedance);
       if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
       }
       else  {hpe1445_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER4 );}
    
       errStatus = viPrintf(vi,":OUTP:LOAD %s;",load);
       if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
       errStatus = viPrintf(vi,":SOUR:VOLT:LEV:IMM %f\n",ampl);
       if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
       errStatus = viPrintf(vi,"INIT:IMM\n");
       if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
    }
    hpe1445_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1445_squareWave
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This function generates a square wave.  The reference
 *           oscillator source is INTernal1 and the trigger source is
 *           INTernal1. These parameters are set internally by this function.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument handle. In this case VI_NULL.
 * 
 * PARAM 2 : ViReal64 freq
 * IN        
 *            Signal frequency.
 * 
 *      MAX = hpe1445_SQU_FREQ_MAX   42.94e6
 *      MIN = hpe1445_SQU_FREQ_MIN   0
 * 
 * PARAM 3 : ViReal64 ampl
 * IN        
 *            Signal amplitude in volts peak.
 * 
 *      MAX = hpe1445_SQU_AMPL_MAX   12.05
 *      MIN = hpe1445_SQU_AMPL_MIN   0
 * 
 * PARAM 4 : ViString polarity
 * IN        
 *            Signal polarity. "NORM" - initial voltage goes positive.
 *            "INV" - initial voltage goes negative.
 * 
 * PARAM 5 : ViReal64 offset
 * IN        
 *            Signal offset.
 * 
 *      MAX = hpe1445_SQU_OFFSETMAX   12.05
 *      MIN = hpe1445_SQU_OFFSETMIN   -12.05
 * 
 * PARAM 6 : ViInt32 impedance
 * IN        
 *            Sets the output impedance to 50 or 75 ohms.
 * 
 *      Macro Name        Value  Description
 *      -----------------------------------------------------------
 *      hpe1445_Z_50         50  50 Ohm
 *      hpe1445_Z_75         75  75 Ohm
 * 
 * PARAM 7 : ViString load
 * IN        
 *            Output load. "MIN" (50), "MAX" (75) or "INF" (infinity).
 *            Use infinity to get the correct amplitude on an open circuit.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           the "hpe1413_error_message" routine.
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1445_squareWave(
  ViSession vi,
  ViReal64 freq,
  ViReal64 ampl,
  ViString polarity,
  ViReal64 offset,
  ViInt32 impedance,
  ViString load)
{
    ViStatus errStatus = 0;
    struct hpe1445_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ hpe1445_LOG_STATUS( vi, 0, errStatus ); }

    hpe1445_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1445_CDE_INIT( "hpe1445_squareWave" );

    /* Perform Error Checking on Each Parameter */
    hpe1445_CHK_REAL_RANGE(freq
                          ,hpe1445_SQU_FREQ_MIN
                          ,hpe1445_SQU_FREQ_MAX
                          ,VI_ERROR_PARAMETER2);

    hpe1445_CHK_REAL_RANGE(ampl
                          ,hpe1445_SQU_AMPL_MIN
                          ,hpe1445_SQU_AMPL_MAX
                          ,VI_ERROR_PARAMETER3);

    hpe1445_CHK_REAL_RANGE(offset
                          ,hpe1445_SQU_OFFSETMIN
                          ,hpe1445_SQU_OFFSETMAX
                          ,VI_ERROR_PARAMETER5);

    {  
       errStatus = viPrintf(vi,":ABORT\n");
       if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
       errStatus = viPrintf(vi,":VOLT:UNIT V\n;");
       if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
       errStatus = viPrintf(vi,":FREQ1:MODE FIX;");
       if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
       errStatus = viPrintf(vi,":SOUR:ROSC:SOUR INT1;");
       if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
       errStatus = viPrintf(vi,":TRIG:STAR:SOUR INT1;");
       if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
       errStatus = viPrintf(vi,":SOUR:FREQ1:RANG 0;");
       if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
       
       errStatus = viPrintf(vi,":SOUR:FREQ:FIX %f;",freq);
       if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
       errStatus = viPrintf(vi,":SOUR:FUNC:SHAPE SQU;");
       if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
       errStatus = viPrintf(vi,":RAMP:POL %s;",polarity);
       if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
       errStatus = viPrintf(vi,":VOLT:OFFS %f;",offset);
       if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
       if( impedance == 50 || impedance == 75)
       {
       errStatus = viPrintf(vi,":OUTP:IMP %d;",(int)impedance);
       if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
       } else {hpe1445_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER6 );}
    
       errStatus = viPrintf(vi,":OUTP:LOAD %s;",load);
       if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
       errStatus = viPrintf(vi,":SOUR:VOLT:LEV:IMM %f\n",ampl);
       if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
       errStatus = viPrintf(vi,"INIT:IMM\n");
       if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
    }
    hpe1445_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1445_sweep
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This function sets up a sine wave, square wave, or triangle
 *           wave sweep.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument handle. In this case VI_NULL.
 * 
 * PARAM 2 : ViString shape
 * IN        
 *            Waveform swept. Can be 'SIN' (sine wave), 'SQU' (square wave),
 *            or 'TRI' (triangle wave). If TRI, then the points parameter
 *            is used as the number of points in the ramp.
 * 
 * PARAM 3 : ViReal64 startfreq
 * IN        
 *            Sweep start freqency.
 * 
 *      MAX = hpe1445_SWEEPSTART_MAX   42.94e6
 *      MIN = hpe1445_SWEEPSTART_MIN   0
 * 
 * PARAM 4 : ViReal64 stopfreq
 * IN        
 *            Sweep stop frequency.
 * 
 *      MAX = hpe1445_SWEEPSTOP_MAX   42.94e6
 *      MIN = hpe1445_SWEEPSTOP_MIN   0
 * 
 * PARAM 5 : ViReal64 ampl
 * IN        
 *            Amplitude in volts peak.
 * 
 *      MAX = hpe1445_SWEEPAMPL_MAX   12.05
 *      MIN = hpe1445_SWEEPAMPL_MIN   -12.05
 * 
 * PARAM 6 : ViInt32 points
 * IN        
 *            Number of ramp points, used only for triangle sweeps.
 * 
 *      MAX = hpe1445_SWEEPPOINTS_MAX   262144
 *      MIN = hpe1445_SWEEPPOINTS_MIN   4
 * 
 * PARAM 7 : ViInt32 impedance
 * IN        
 *            Sets the output impedance to 50 or 75 ohms.
 * 
 *      Macro Name        Value  Description
 *      -----------------------------------------------------------
 *      hpe1445_Z_50         50  50 Ohm
 *      hpe1445_Z_75         75  75 Ohm
 * 
 * PARAM 8 : ViString load
 * IN        
 *            Set the output load. 'MIN' (50), 'MAX' (75), or 'INF'
 *           (infinity).
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           the "hpe1413_error_message" routine.
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1445_sweep(
  ViSession vi,
  ViString shape,
  ViReal64 startfreq,
  ViReal64 stopfreq,
  ViReal64 ampl,
  ViInt32 points,
  ViInt32 impedance,
  ViString load)
{
    ViStatus errStatus = 0;
    struct hpe1445_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ hpe1445_LOG_STATUS( vi, 0, errStatus ); }

    hpe1445_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1445_CDE_INIT( "hpe1445_sweep" );

    /* Perform Error Checking on Each Parameter */
    hpe1445_CHK_REAL_RANGE(startfreq
                          ,hpe1445_SWEEPSTART_MIN
                          ,hpe1445_SWEEPSTART_MAX
                          ,VI_ERROR_PARAMETER3);

    hpe1445_CHK_REAL_RANGE(stopfreq
                          ,hpe1445_SWEEPSTOP_MIN
                          ,hpe1445_SWEEPSTOP_MAX
                          ,VI_ERROR_PARAMETER4);

    hpe1445_CHK_REAL_RANGE(ampl
                          ,hpe1445_SWEEPAMPL_MIN
                          ,hpe1445_SWEEPAMPL_MAX
                          ,VI_ERROR_PARAMETER5);

    hpe1445_CHK_LONG_RANGE(points
                          ,hpe1445_SWEEPPOINTS_MIN
                          ,hpe1445_SWEEPPOINTS_MAX
                          ,VI_ERROR_PARAMETER6);

    { 
      
      errStatus = viPrintf(vi,":ABORT\n");
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
      errStatus = viPrintf(vi,":VOLT:UNIT V\n;");
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
      errStatus = viPrintf(vi,":SOUR:ROSC:SOUR INT1;");
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
      errStatus = viPrintf(vi,":TRIG:STAR:SOUR INT1;");
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
      errStatus = viPrintf(vi,":SOUR:FREQ1:RANG 0;");
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
      errStatus = viPrintf(vi,":SOUR:FREQ1:STAR %g;",startfreq);
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
      errStatus = viPrintf(vi,":SOUR:FREQ1:STOP %g;",stopfreq);
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
      errStatus = viPrintf(vi,":SOUR:SWE:COUNT INF;");
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
      if( strcmp(shape,"TRI") == 0)
      {
    	errStatus = viPrintf(vi,":SOUR:RAMP:POINTS %d;",(int)points);
    	if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
      }
    
      if( strcmp(shape,"SQU") == 0 || strcmp(shape,"TRI")==0)
      {
    	errStatus = viPrintf(vi,":VOLT:OFFS 0;");
    	if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
    	errStatus = viPrintf(vi,":RAMP:POL NORM;");
    	if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
      }
    
      if( impedance == 50 || impedance == 75)
      {
      errStatus = viPrintf(vi,":OUTP:IMP %d;",(int)impedance);
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
      }else {hpe1445_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER7 );}
     
      errStatus = viPrintf(vi,":OUTP:LOAD %s;",load);
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
      errStatus = viPrintf(vi,":SOUR:FUNC:SHAP %s;",shape);
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
     
      errStatus = viPrintf(vi,":SOUR:FREQ1:MODE SWE;");
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
      errStatus = viPrintf(vi,":SOUR:VOLT:LEV:IMM:AMPL %g\n",ampl);
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
      
    
      errStatus = viPrintf(vi,":INIT:IMM\n");
      if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
    }
    hpe1445_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1445_triWave
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This function generates a triangle wave. The reference
 *           oscillator source is INTernal1 and the trigger source is
 *           INTernal1. These parameters are set internally by this function.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument handle. In this case VI_NULL.
 * 
 * PARAM 2 : ViReal64 freq
 * IN        
 *            Signal frequency.
 * 
 *      MAX = hpe1445_TRI_FREQ_MAX   42.94e6
 *      MIN = hpe1445_TRI_FREQ_MIN   0
 * 
 * PARAM 3 : ViReal64 ampl
 * IN        
 *            Signal amplitude in volts peak.
 * 
 *      MAX = hpe1445_TRI_AMPL_MAX   12.05
 *      MIN = hpe1445_TRI_AMPL_MIN   0
 * 
 * PARAM 4 : ViString polarity
 * IN        
 *            Signal polarity. "NORM" - initial voltage goes positive.
 *            "INV" - initial voltage goes negative.
 * 
 * PARAM 5 : ViReal64 offset
 * IN        
 *            Signal offset.
 * 
 *      MAX = hpe1445_TRI_OFFSETMAX   12.05
 *      MIN = hpe1445_TRI_OFFSETMIN   -12.05
 * 
 * PARAM 6 : ViInt32 impedance
 * IN        
 *            Sets the output impedance to 50 or 75 ohms.
 * 
 *      Macro Name        Value  Description
 *      -----------------------------------------------------------
 *      hpe1445_Z_50         50  50 Ohm
 *      hpe1445_Z_75         75  75 Ohm
 * 
 * PARAM 7 : ViString load
 * IN        
 *            Output load. "MIN" (50), "MAX" (75) or "INF" (infinity).
 *            Use infinity to get the correct amplitude on an open circuit.
 * 
 * PARAM 8 : ViInt32 points
 * IN        
 *            Set the number of ramp points.
 * 
 *      MAX = hpe1445_TRI_POINTSMAX   262144
 *      MIN = hpe1445_TRI_POINTSMIN   4
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           the "hpe1413_error_message" routine.
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1445_triWave(
  ViSession vi,
  ViReal64 freq,
  ViReal64 ampl,
  ViString polarity,
  ViReal64 offset,
  ViInt32 impedance,
  ViString load,
  ViInt32 points)
{
    ViStatus errStatus = 0;
    struct hpe1445_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ hpe1445_LOG_STATUS( vi, 0, errStatus ); }

    hpe1445_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1445_CDE_INIT( "hpe1445_triWave" );

    /* Perform Error Checking on Each Parameter */
    hpe1445_CHK_REAL_RANGE(freq
                          ,hpe1445_TRI_FREQ_MIN
                          ,hpe1445_TRI_FREQ_MAX
                          ,VI_ERROR_PARAMETER2);

    hpe1445_CHK_REAL_RANGE(ampl
                          ,hpe1445_TRI_AMPL_MIN
                          ,hpe1445_TRI_AMPL_MAX
                          ,VI_ERROR_PARAMETER3);

    hpe1445_CHK_REAL_RANGE(offset
                          ,hpe1445_TRI_OFFSETMIN
                          ,hpe1445_TRI_OFFSETMAX
                          ,VI_ERROR_PARAMETER5);

    hpe1445_CHK_LONG_RANGE(points
                          ,hpe1445_TRI_POINTSMIN
                          ,hpe1445_TRI_POINTSMAX
                          ,VI_ERROR_PARAMETER8);

    {  
       errStatus = viPrintf(vi,":ABORT\n");
       if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
       errStatus = viPrintf(vi,":VOLT:UNIT V\n;");
       if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
       errStatus = viPrintf(vi,":FREQ1:MODE FIX;");
       if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
       errStatus = viPrintf(vi,":SOUR:ROSC:SOUR INT1;");
       if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
       errStatus = viPrintf(vi,":TRIG:STAR:SOUR INT1;");
       if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
       errStatus = viPrintf(vi,":SOUR:FREQ1:RANG 0;");
       if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
       errStatus = viPrintf(vi,":SOUR:FREQ:FIX %f;",freq);
       if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
       errStatus = viPrintf(vi,":SOUR:FUNC:SHAPE TRI;");
       if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
       errStatus = viPrintf(vi,":RAMP:POL %s;",polarity);
       if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
       errStatus = viPrintf(vi,":VOLT:OFFS %f;",offset);
       if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
       if( impedance == 50 || impedance == 75)
       {
       errStatus = viPrintf(vi,":OUTP:IMP %d;",(int)impedance);
       if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
       } else {hpe1445_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER6 );}
    
       errStatus = viPrintf(vi,":OUTP:LOAD %s;",load);
       if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
       errStatus = viPrintf(vi,":SOUR:VOLT:LEV:IMM %f\n",ampl);
       if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
       errStatus = viPrintf(vi,"INIT:IMM\n");
       if (errStatus < VI_SUCCESS){hpe1445_LOG_STATUS( vi, thisPtr, errStatus );}
    
    }
    hpe1445_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}
